<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Crefopay\Action\HostedPageBefore\Api;

use Lifestyle\Sylius\Crefopay\Request\HostedPageBefore\Api\CreateTransaction;
use Payum\Core\Action\ActionInterface;
use Payum\Core\ApiAwareInterface;
use Payum\Core\Bridge\Spl\ArrayObject;
use Payum\Core\Exception\RequestNotSupportedException;
use Payum\Core\GatewayAwareInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Upg\Library\Request\CreateTransaction as UpgCreateTransactionRequest;
use Upg\Library\User\Type as UpgUserType;

/**
 * Class CreateTransactionAction
 *
 * @package Lifestyle\Sylius\Crefopay\Action\HostedPageBefore
 */
class CreateTransactionAction extends BaseApiAwareAction implements ActionInterface, GatewayAwareInterface, ApiAwareInterface
{
    /**
     * {@inheritdoc}
     *
     * @param $request CreateTransaction
     */
    public function execute($request)
    {
        RequestNotSupportedException::assertSupports($this, $request);
        $details = ArrayObject::ensureArrayObject($request->getModel());

        try {
            /** @var OrderInterface $order */
            $order = $request->getFirstModel()->getOrder();

            if (false === $this->isTransactionExisting($order)) {
                $createTransactionRequest = $this->api->getRequestModelFactory()
                    ->createTransaction()
                    ->setMerchantReference(substr($request->getToken()->getHash(), 0, 30))
                    ->setOrderID($order->getId())
                    ->setUserID($order->getCustomer()->getId())
                    ->setIntegrationType(UpgCreateTransactionRequest::INTEGRATION_TYPE_HOSTED_BEFORE)
                    ->setContext(UpgCreateTransactionRequest::CONTEXT_ONLINE)
                    ->setUserType(UpgUserType::USER_TYPE_PRIVATE)
                    ->setUserData(
                        $this->api->getRequestModelFactory()->person()
                            ->setName($order->getCustomer()->getFirstName())
                            ->setSurname($order->getCustomer()->getLastName())
                            ->setEmail($order->getCustomer()->getEmail())
                    )
                    ->setBillingAddress(
                        $this->api->getRequestModelFactory()->address()
                            ->setStreet($order->getBillingAddress()->getStreet())
                            ->setZip($order->getBillingAddress()->getPostcode())
                            ->setCity($order->getBillingAddress()->getCity())
                            ->setCountry($order->getBillingAddress()->getCountryCode())
                    )
                    ->setAmount(
                        $this->api->getRequestModelFactory()->amount()
                            ->setAmount($order->getTotal())
                    )
                    ->setLocale(strtoupper($order->getLocaleCode()))
                ;

                $this->api->addMacToRequest($createTransactionRequest, $order->getId());

                foreach ($order->getItems() as $orderItem) {
                    $createTransactionRequest->addBasketItem(
                        $this->api->getRequestModelFactory()->basketItem()
                            ->setBasketItemText($orderItem->getProductName())
                            ->setBasketItemCount($orderItem->getQuantity())
                            ->setBasketItemAmount(
                                $this->api->getRequestModelFactory()->amount()
                                    ->setAmount($orderItem->getSubtotal())
                            )
                    );
                }

                if (strlen($order->getCustomer()->getCompanyName()) > 0) {
                    $createTransactionRequest
                        ->setUserType(UpgUserType::USER_TYPE_BUSINESS)
                        ->setCompanyData(
                            $this->api->getRequestModelFactory()->company()
                                ->setCompanyName($order->getCustomer()->getCompanyName())
                                ->setEmail($order->getCustomer()->getEmail())
                        )
                    ;
                }

                $response = $this->api->getApiFactory()->createTransaction($createTransactionRequest)->sendRequest();

                // set paymentUrl to payment details
                $details['paymentFrameUrl'] = $response->getData('redirectUrl');
            }
        } catch (\Exception $exception) {
            $detailMessages = $details->get('messages', []);
            $detailMessages[__CLASS__ . '::' . __FUNCTION__] = $exception->getMessage();
            $details->defaults([
                'messages' => $detailMessages
            ]);
        }
    }

    /**
     * @param OrderInterface $order
     *
     * @return bool
     */
    public function isTransactionExisting(OrderInterface $order): bool
    {
        $status = true;

        try {
            // check if we have an active transaction
            $getTransactionStatusRequest = $this->api->getRequestModelFactory()
                ->transactionStatus()
                ->setOrderID($order->getId());

            $this->api->addMacToRequest($getTransactionStatusRequest, $order->getId());

            $response = $this->api->getApiFactory()->transactionStatus($getTransactionStatusRequest)->sendRequest();
            if (empty($response->getData('transactionStatus'))) {
                $status = false;
            }
        } catch (\Exception $exception) {
            $status = false;
        }

        return $status;
    }

    /**
     * {@inheritdoc}
     */
    public function supports($request)
    {
        return
            $request instanceof CreateTransaction &&
            $request->getModel() instanceof \ArrayAccess
        ;
    }
}
