<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Crefopay\Action\HostedPageBefore\Api;

use Lifestyle\Sylius\Crefopay\Request\HostedPageBefore\Api\ReserveTransaction;
use Payum\Core\Action\ActionInterface;
use Payum\Core\ApiAwareInterface;
use Payum\Core\Bridge\Spl\ArrayObject;
use Payum\Core\Exception\RequestNotSupportedException;
use Payum\Core\GatewayAwareInterface;
use Payum\Core\Reply\HttpRedirect;
use Sylius\Component\Core\Model\OrderInterface;

/**
 * Class ReserveTransactionAction
 *
 * @package Lifestyle\Sylius\Crefopay\Action\HostedPageBefore\Api
 */
class ReserveTransactionAction extends BaseApiAwareAction implements ActionInterface, GatewayAwareInterface, ApiAwareInterface
{
    /**
     * @param ReserveTransaction $request
     */
    public function execute($request)
    {
        RequestNotSupportedException::assertSupports($this, $request);
        $details = ArrayObject::ensureArrayObject($request->getModel());

        // only start reservation if not yet done
        if (false === $details->get('reserved', false)) {
            $redirectUrl = '';
            try {
                /** @var OrderInterface $order */
                $order = $request->getFirstModel()->getOrder();

                $reserveTransactionRequest = $this->api->getRequestModelFactory()
                    ->reserve()
                    ->setOrderID($order->getId())
                    ->setPaymentMethod($details['paymentMethod'])
                ;

                if (null !== $details->get('paymentInstrumentID', null)) {
                    $reserveTransactionRequest->setPaymentInstrumentID($details->get('paymentInstrumentID'));
                }

                $this->api->addMacToRequest($reserveTransactionRequest, $order->getId());

                $response = $this->api->getApiFactory()->reserve($reserveTransactionRequest)->sendRequest();
                if (in_array($response->getData('resultCode'), [0, 1])) {
                    $details['reserved'] = true;
                    // check for additionalData on some payment types
                    if (is_array($response->getData('additionalData'))) {
                        $details->defaults([
                            'additionalData' => array_merge(
                                $details->get('additionalData', []),
                                $response->getData('additionalData')
                            )
                        ]);
                    }
                    if (1 === $response->getData('resultCode')) {
                        $redirectUrl = $response->getData('redirectUrl');
                    }
                }
            } catch (\Exception $exception) {
                $detailMessages = $details->get('messages', []);
                $detailMessages[__CLASS__ . '::' . __FUNCTION__] = $exception->getMessage();
                $details->defaults([
                    'messages' => $detailMessages
                ]);
            }

            // redirect to external payment url if needed
            if (strlen($redirectUrl) > 0) {
                // invalidate the capture token
                $this->api->getPayum()->getHttpRequestVerifier()->invalidate($request->getToken());

                throw new HttpRedirect($redirectUrl);
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function supports($request)
    {
        return
            $request instanceof ReserveTransaction &&
            $request->getModel() instanceof \ArrayAccess
        ;
    }
}
