<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Crefopay\Api;

use Payum\Core\Payum;
use Sylius\Component\Order\Repository\OrderRepositoryInterface;
use Symfony\Component\Routing\RouterInterface;
use Upg\Library\Callback\MacCalculator;
use Upg\Library\Config;
use Upg\Library\Request\AbstractRequest;

/**
 * Class CrefopayApi
 *
 * @package Lifestyle\Sylius\Crefopay\Api
 */
class CrefopayApi
{
    public const PRODUCTION_API = 'https://api.crefopay.de/2.0/';
    public const SANDBOX_API = 'https://sandbox.crefopay.de/2.0/';

    /**
     * @var Config
     */
    protected $config;
    /**
     * @var CrefopayApiRequestModelFactory
     */
    protected $requestModelFactory;
    /**
     * @var CrefopayApiFactory
     */
    protected $apiFactory;
    /**
     * @var RouterInterface
     */
    protected $router;
    /**
     * @var Payum
     */
    protected $payum;

    /**
     * @var OrderRepositoryInterface
     */
    protected $orderRepository;

    /**
     * CrefopayApi constructor.
     *
     * @param Config                         $config
     * @param CrefopayApiFactory             $apiFactory
     * @param CrefopayApiRequestModelFactory $requestModelFactory
     * @param RouterInterface                $router
     * @param Payum                          $payum
     * @param OrderRepositoryInterface       $orderRepository
     */
    public function __construct(
        Config $config,
        CrefopayApiFactory $apiFactory,
        CrefopayApiRequestModelFactory $requestModelFactory,
        RouterInterface $router,
        Payum $payum,
        OrderRepositoryInterface $orderRepository
    ) {
        $this->config = $config;
        $this->apiFactory = $apiFactory;
        $this->requestModelFactory = $requestModelFactory;
        $this->router = $router;
        $this->payum = $payum;
        $this->orderRepository = $orderRepository;
    }

    /**
     * @return Config
     */
    public function getConfig(): Config
    {
        return $this->config;
    }

    /**
     * @return CrefopayApiRequestModelFactory
     */
    public function getRequestModelFactory(): CrefopayApiRequestModelFactory
    {
        return $this->requestModelFactory;
    }

    /**
     * @return CrefopayApiFactory
     */
    public function getApiFactory(): CrefopayApiFactory
    {
        return $this->apiFactory;
    }

    /**
     * @return RouterInterface
     */
    public function getRouter(): RouterInterface
    {
        return $this->router;
    }

    /**
     * @return Payum
     */
    public function getPayum(): Payum
    {
        return $this->payum;
    }

    /**
     * @return OrderRepositoryInterface
     */
    public function getOrderRepository(): OrderRepositoryInterface
    {
        return $this->orderRepository;
    }

    /**
     * @param AbstractRequest $request
     * @param int             $orderId
     */
    public function addMacToRequest(AbstractRequest $request, int $orderId): void
    {
        $request->setMac(
            (new MacCalculator(
                $this->config,
                [
                    0 => $this->config->getMerchantID(),
                    1 => $orderId,
                    2 => $this->config->getStoreID(),
                ]
            ))->calculateMac()
        );
    }
}
