<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Crefopay\Command;

use App\Entity\Order;
use Doctrine\ORM\EntityManager as NotificationManager;
use Lifestyle\Sylius\Crefopay\Entity\CrefopayNotification;
use Lifestyle\Sylius\Crefopay\Request\HostedPageBefore\Api\CaptureLateTransaction;
use Payum\Core\Payum;
use Psr\Log\LoggerInterface;
use Sylius\Bundle\PayumBundle\Factory\GetStatusFactoryInterface;
use Sylius\Bundle\ResourceBundle\Doctrine\ORM\EntityRepository as NotificationRepository;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Command\LockableTrait;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class CrefopayNotifyCommand
 *
 * @package Lifestyle\Sylius\Crefopay\Command
 */
class CrefopayNotificationProcessorCommand extends Command
{
    const GATEWAY_NAME = 'crefopay_hostedbefore';
    const LOCK_ID = 'crefopayProcessNotifications';

    use LockableTrait;

    /**
     * @var string
     */
    protected static $defaultName = 'crefopay:process:notifications';
    /**
     * @var Payum
     */
    protected $payum;
    /**
     * @var NotificationRepository
     */
    protected $notificationRepository;
    /**
     * @var NotificationManager
     */
    protected $notificationManager;
    /**
     * @var GetStatusFactoryInterface
     */
    protected $statusFactory;
    /**
     * @var LoggerInterface
     */
    protected $notificationLogger;

    /**
     * CrefopayNotificationProcessorCommand constructor.
     *
     * @param Payum                     $payum
     * @param NotificationRepository    $notificationRepository
     * @param NotificationManager       $notificationManager
     * @param GetStatusFactoryInterface $statusFactory
     * @param LoggerInterface           $notificationLogger
     */
    public function __construct(
        Payum $payum,
        NotificationRepository $notificationRepository,
        NotificationManager $notificationManager,
        GetStatusFactoryInterface $statusFactory,
        LoggerInterface $notificationLogger
    ) {
        $this->payum = $payum;
        $this->notificationRepository = $notificationRepository;
        $this->notificationManager = $notificationManager;
        $this->statusFactory = $statusFactory;
        $this->notificationLogger = $notificationLogger;

        parent::__construct();
    }

    /**
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        if ($this->lock(self::LOCK_ID)) {
            /** @var CrefopayNotification[] $notifications */
            $notifications = $this->notificationRepository->findBy(['processed' => false]);

            foreach ($notifications as $notification) {
                try {
                    if (strlen($notification->getErrorMessage()) === 0) {
                        $token = $this->payum->getTokenStorage()->find($notification->getTokenHash());
                        // add values to token details
                        $token->setUrlParameters([
                            'transactionStatus' => $notification->getTransactionStatus(),
                            'paymentStatus' => $notification->getOrderStatus(),
                        ]);

                        // run capture late transactoion
                        $this->payum->getGateway(self::GATEWAY_NAME)->execute(new CaptureLateTransaction($token));
                        // run status
                        $this->payum->getGateway(self::GATEWAY_NAME)->execute($this->statusFactory->createNewWithModel($token));
                    }

                    // mark as processed
                    $notification->setProcessed(true);
                    $this->notificationManager->persist($notification);
                    $this->notificationManager->flush();

                    // invalidate token
                    $this->payum->getHttpRequestVerifier()->invalidate($token);
                } catch (\Exception $exception) {
                    $this->notificationLogger->error($exception->getMessage());
                }
            }

            // release lock
            $this->release();
        }
    }
}
