<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Sylius\DataModelPlugin\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use LifeStyle\Sylius\DataModelPlugin\Entity\AssetStorage;
use LifeStyle\Sylius\DataModelPlugin\Entity\AssetStorageInterface;
use LifeStyle\Sylius\PricingPlugin\Traits\CustomDiscountableTrait;
use LifeStyle\Sylius\PricingPlugin\Traits\CustomDiscountableInterface;
use LifeStyle\Sylius\PricingPlugin\Traits\TierPriceableTrait;
use LifeStyle\Sylius\PricingPlugin\Traits\TierPriceableInterface;
use Sylius\Component\Core\Model\ProductVariant as BaseProductVariant;
use Sylius\Component\Resource\Model\ToggleableTrait;
use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\Criteria;

/**
 * Class ProductVariant
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    LifeStyle\Sylius\DataModelPlugin\Entity
 */
class ProductVariant extends BaseProductVariant implements ProductVariantInterface, TierPriceableInterface, CustomDiscountableInterface
{
    // Brings in a new function to disable product-variants
    use ToggleableTrait;
    use TierPriceableTrait;
    use CustomDiscountableTrait;


    /**
     * @var string|null
     */
    private $sku;

    /**
     * @var AssetStorage[]|ArrayCollection|null
     */
    protected $assets;

    /**
     * @var string|null
     */
    protected $concept;

    /**
     * @var string|null
     */
    protected $availabilityIndicator;

    /**
     * ProductVariant constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->initTierPriceableTrait();
        $this->initCustomDiscountableTrait();
        $this->assets = new ArrayCollection();
    }

    /**
     * {@inheritdoc}
     */
    public function getOptionValues(): Collection
    {
        return ($this->optionValues->isEmpty()) ? $this->optionValues : $this->optionValues->matching((Criteria::create())->orderBy(['position' => Criteria::ASC]));
    }

    /**
     * @return Collection
     */
    public function getAssets(): Collection
    {
        return $this->assets;
    }

    /**
     * @param AssetStorageInterface $assetStorage
     */
    public function addAsset(AssetStorageInterface $assetStorage): void
    {
        if (!$this->hasAsset($assetStorage)) {
            $this->assets->add($assetStorage);
        }
    }

    /**
     * @param AssetStorageInterface $assetStorage
     */
    public function removeAsset(AssetStorageInterface $assetStorage): void
    {
        if ($this->hasAsset($assetStorage)) {
            $this->assets->removeElement($assetStorage);
        }
    }

    /**
     * @param AssetStorageInterface $assetStorage
     * @return bool
     */
    public function hasAsset(AssetStorageInterface $assetStorage): bool
    {
        return $this->assets->contains($assetStorage);
    }

    /**
     * @return bool
     */
    public function hasAssets()
    {
        return $this->assets->count() !== 0 ? true : false;
    }

    /**
     * @return string|null
     */
    public function getSku(): ?string
    {
        return $this->sku;
    }

    /**
     * @param string|null $sku
     */
    public function setSku(?string $sku): void
    {
        $this->sku = $sku;
    }

    /**
     * @return null|string
     */
    public function getAvailabilityIndicator(): ?string
    {
        return $this->availabilityIndicator;
    }

    /**
     * @param null|string $availabilityIndicator
     *
     * @return ProductVariant
     */
    public function setAvailabilityIndicator(?string $availabilityIndicator): ProductVariant
    {
        $this->availabilityIndicator = $availabilityIndicator;

        return $this;
    }

    /**
     * @return null|string
     */
    public function getConcept(): ?string
    {
        return $this->concept;
    }

    /**
     * @param null|string $concept
     *
     * @return ProductVariant
     */
    public function setConcept(?string $concept): ProductVariant
    {
        $this->concept = $concept;

        return $this;
    }
}
