<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Klarna\Action;

use Lifestyle\Sylius\Klarna\Request\Api\CreateSession;
use Lifestyle\Sylius\Klarna\Request\Api\PlaceOrder;
use Payum\Core\Action\ActionInterface;
use Payum\Core\Bridge\Spl\ArrayObject;
use Payum\Core\GatewayAwareInterface;
use Payum\Core\GatewayAwareTrait;
use Payum\Core\Reply\HttpResponse;
use Payum\Core\Request\Authorize;
use Payum\Core\Exception\RequestNotSupportedException;
use Payum\Core\Request\GetHttpRequest;
use Payum\Core\Request\RenderTemplate;

/**
 * Class AuthorizeAction
 *
 * @package Lifestyle\Sylius\Klarna\Action
 */
class AuthorizeAction implements ActionInterface, GatewayAwareInterface
{
    use GatewayAwareTrait;

    /**
     * @var string
     */
    protected $templateName;

    /**
     * @var ArrayObject
     */
    private $config;

    /**
     * AuthorizeAction constructor.
     *
     * @param string      $templateName
     * @param ArrayObject $config
     */
    public function __construct(string $templateName, ArrayObject $config)
    {
        $this->templateName = $templateName;
        $this->config = $config;
    }

    /**
     * {@inheritDoc}
     *
     * @param Authorize $request
     */
    public function execute($request)
    {
        RequestNotSupportedException::assertSupports($this, $request);
        $model = ArrayObject::ensureArrayObject($request->getModel());
        $model['state'] = 'none';
        $this->gateway->execute($httpRequest = new GetHttpRequest());
        if (isset($httpRequest->query['state'])) {
            $model['state'] = $httpRequest->query['state'];
        }

        $render = false;
        switch ($model['state']) {
            case "confirm":
                $model['authToken'] = $httpRequest->query['authToken'];

                $placeOrderRequest = new PlaceOrder($request->getToken());
                $placeOrderRequest->setModel($request->getFirstModel());
                $placeOrderRequest->setModel($request->getModel());

                $this->gateway->execute($placeOrderRequest);
                break;
            default:
                $createSessionRequest = new CreateSession($request->getToken());
                $createSessionRequest->setModel($request->getFirstModel());
                $createSessionRequest->setModel($request->getModel());

                $this->gateway->execute($createSessionRequest);

                $render = true;
                break;
        }

        if (true === $render) {
            $renderTemplate = new RenderTemplate($this->templateName, [
                'order' => $request->getFirstModel()->getOrder(),
                'clientToken' => $model['clientToken'],
                'paymentMethods' => $this->filterAndSortPaymenMethods($model['paymentMethods']),
            ]);
            $this->gateway->execute($renderTemplate);

            throw new HttpResponse($renderTemplate->getResult());
        }
    }

    /**
     * @param array|null $paymentMenthods
     * @return array
     */
    private function filterAndSortPaymenMethods(?array $paymentMenthods) {
        $activatedPaymentMethods = [];

        if (null == $paymentMenthods) {
            return $activatedPaymentMethods;
        }

        foreach ($paymentMenthods as $paymentMethod) {
            if (true !== $this->config['disable_' . $paymentMethod['identifier']]) {
                $activatedPaymentMethods[] = $paymentMethod;
            }
        }

        usort($activatedPaymentMethods, function ($first, $second) {
            return strcmp ($first['identifier'], $second['identifier']);
        });

        return $activatedPaymentMethods;
    }

    /**
     * {@inheritDoc}
     */
    public function supports($request)
    {
        return
            $request instanceof Authorize &&
            $request->getModel() instanceof \ArrayAccess
        ;
    }
}
