<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author    Franz Weisflug
 * @copyright 2019 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Lifestyle\Sylius\Klarna\Service;

use Sylius\Component\Core\Model\AddressInterface;
use Sylius\Component\Core\Model\CustomerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Shipping\Calculator\DelegatingCalculatorInterface;
use Sylius\Component\Taxation\Calculator\CalculatorInterface;
use Sylius\Component\Taxation\Resolver\TaxRateResolverInterface;

class KlarnaRequest
{
    /**
     * @var TaxRateResolverInterface
     */
    protected $taxRateResolver;

    /**
     * @var CalculatorInterface
     */
    protected $taxCalculator;

    /**
     * @var DelegatingCalculatorInterface
     */
    protected $shippingCalculator;

    /**
     * KlarnaRequest constructor.
     *
     * @param TaxRateResolverInterface $taxRateResolver
     * @param CalculatorInterface $taxCalculator
     * @param DelegatingCalculatorInterface $shippingCalculator
     */
    public function __construct(
        TaxRateResolverInterface $taxRateResolver,
        CalculatorInterface $taxCalculator,
        DelegatingCalculatorInterface $shippingCalculator
    ){
        $this->taxRateResolver = $taxRateResolver;
        $this->taxCalculator = $taxCalculator;
        $this->shippingCalculator = $shippingCalculator;
    }

    /**
     * @param OrderInterface $order
     *
     * @return array
     */
    public function generateOrderLineItems(OrderInterface $order)
    {
        $lineItems = [];

        foreach ($order->getItems() as $orderItem) {
            $taxRate = $this->taxRateResolver->resolve($orderItem->getVariant());
            $taxRateValue = intval($taxRate->getAmountAsPercentage() * 100);
            $unitPriceWithTax = $orderItem->getUnitPrice() + $this->taxCalculator->calculate($orderItem->getUnitPrice(), $taxRate);
            $lineItems[] = [
                'type' => 'physical',
                'reference' => $orderItem->getProduct()->getId(),
                'name' => $orderItem->getProductName(),
                'quantity' => $orderItem->getQuantity(),
                'quantity_unit' => 'ST',
                'unit_price' => $taxRate->isIncludedInPrice() ? $orderItem->getUnitPrice() : $unitPriceWithTax,
                'tax_rate' => $taxRateValue,
                'total_amount' => $orderItem->getTotal(),
                'total_tax_amount' => $orderItem->getTaxTotal(),
            ];
        }

        foreach ($order->getShipments() as $shipment) {
            $taxRate = $this->taxRateResolver->resolve($shipment->getMethod());
            $taxRateValue = intval($taxRate->getAmountAsPercentage() * 100);
            $shippingCosts = $this->shippingCalculator->calculate($shipment);
            $shippingCostsTax = $this->taxCalculator->calculate($shippingCosts, $taxRate);
            $shippingCostsWithTax = $shippingCosts + $shippingCostsTax;
            $shippingCostsReal = $taxRate->isIncludedInPrice() ? $shippingCosts : $shippingCostsWithTax;

            $lineItems[] = [
                'type' => 'shipping_fee',
                'name' => 'Shipping',
                'quantity' => 1,
                'unit_price' => $shippingCostsReal,
                'tax_rate' => $taxRateValue,
                'total_amount' => $shippingCostsReal,
                'total_tax_amount' => $shippingCostsTax,
            ];
        }

        return $lineItems;
    }

    /**
     * @param null|AddressInterface $address
     * @param null|CustomerInterface $customer
     *
     * @return array|null
     */
    public function generateAddress(?AddressInterface $address, ?CustomerInterface $customer)
    {
        if (null === $address) {
            return null;
        }

        return [
            'given_name' => $address->getFirstName(),
            'family_name' => $address->getLastName(),
            'email' => null === $customer ? null : $customer->getEmail(),
            'street_address' => $address->getStreet(),
            'postal_code' => $address->getPostcode(),
            'city' => $address->getCity(),
            'region' => $address->getProvinceCode(),
            'phone' => $address->getPhoneNumber(),
            'country' => $address->getCountryCode()
        ];
    }

    /**
     * @param OrderInterface $order
     *
     * @return array
     */
    public function generateShippingInfo(OrderInterface $order)
    {
        $shippingInfo = [];

        foreach ($order->getShipments() as $shipment) {
            $shippingInfo[] = [
                'shipping_company' => $shipment->getMethod()->getName(),
                'shipping_method' => 'Home',
            ];
        }

        return $shippingInfo;
    }

    /**
     * @param string $locale
     *
     * @return mixed|string
     */
    public function generateLocale(string $locale)
    {
        $klarnaLocale = $locale;

        if (2 === strlen($locale)) {
            $klarnaLocale = strtolower($locale) . '-' . strtoupper($locale);
        }

        $klarnaLocale = str_replace('_', '-', $klarnaLocale);
        return $klarnaLocale;
    }
}
