<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author    Franz Weisflug
 * @copyright 2019 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Lifestyle\Sylius\Klarna\Service;

use Sylius\Component\Core\Model\AddressInterface;
use Sylius\Component\Core\Model\AdjustmentInterface;
use Sylius\Component\Core\Model\CustomerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\OrderItemInterface;
use Sylius\Component\Shipping\Calculator\DelegatingCalculatorInterface;
use Sylius\Component\Taxation\Calculator\CalculatorInterface;
use Sylius\Component\Taxation\Resolver\TaxRateResolverInterface;
use Symfony\Component\Intl\Locales;

/**
 * Class KlarnaRequest
 *
 * @package Lifestyle\Sylius\Klarna\Service
 */
class KlarnaRequest
{
    /**
     * @var TaxRateResolverInterface
     */
    protected $taxRateResolver;

    /**
     * @var CalculatorInterface
     */
    protected $taxCalculator;

    /**
     * @var DelegatingCalculatorInterface
     */
    protected $shippingCalculator;

    /**
     * KlarnaRequest constructor.
     *
     * @param TaxRateResolverInterface $taxRateResolver
     * @param CalculatorInterface $taxCalculator
     * @param DelegatingCalculatorInterface $shippingCalculator
     */
    public function __construct(
        TaxRateResolverInterface $taxRateResolver,
        CalculatorInterface $taxCalculator,
        DelegatingCalculatorInterface $shippingCalculator
    ){
        $this->taxRateResolver = $taxRateResolver;
        $this->taxCalculator = $taxCalculator;
        $this->shippingCalculator = $shippingCalculator;
    }

    /**
     * @param OrderInterface $order
     * @param array          $sessionRequestData
     *
     * @return array
     */
    public function generateOrder(OrderInterface $order, array $sessionRequestData): array
    {
        $sessionRequestData['order_amount'] = $order->getTotal();
        $sessionRequestData['order_tax_amount'] = $order->getTaxTotal();

        return $sessionRequestData;
    }

    /**
     * @param OrderInterface $order
     *
     * @return array
     */
    public function generateOrderLineItems(OrderInterface $order)
    {
        $lineItems = [];

        foreach ($order->getItems() as $orderItem) {
            $taxRateValue = 0;
            $unitPrice = $orderItem->getUnitPrice();
            $discount = $this->getUnitDiscount($orderItem);

            $taxRate = $this->taxRateResolver->resolve($orderItem->getVariant());
            if (null !== $taxRate) {
                $taxRateValue = intval($taxRate->getAmountAsPercentage() * 100);
                if (!$taxRate->isIncludedInPrice()) {
                    $unitPrice = $orderItem->getUnitPrice() + $this->taxCalculator->calculate($orderItem->getUnitPrice(), $taxRate);
                    $discount = $discount + $this->taxCalculator->calculate($discount, $taxRate);
                }
            }

            $lineItems[] = [
                'type' => 'physical',
                'reference' => $orderItem->getProduct()->getId(),
                'name' => $orderItem->getProductName(),
                'quantity' => $orderItem->getQuantity(),
                'quantity_unit' => 'ST',
                'unit_price' => $unitPrice,
                'tax_rate' => $taxRateValue,
                'total_amount' => $orderItem->getTotal(),
                'total_tax_amount' => $orderItem->getTaxTotal(),
                'total_discount_amount' => $discount,
            ];
        }

        foreach ($order->getShipments() as $shipment) {
            $taxRateValue = 0;
            $shippingCostsTax = 0;
            $shippingCosts = $this->shippingCalculator->calculate($shipment);
            $shippingCostsReal = $shippingCosts;

            $taxRate = $this->taxRateResolver->resolve($shipment->getMethod());
            if (null !== $taxRate) {
                $taxRateValue = intval($taxRate->getAmountAsPercentage() * 100);
                $shippingCostsTax = $this->taxCalculator->calculate($shippingCosts, $taxRate);

                if (!$taxRate->isIncludedInPrice()) {
                    $shippingCostsReal = $shippingCosts + $shippingCostsTax;
                }
            }

            $lineItems[] = [
                'type' => 'shipping_fee',
                'name' => 'Shipping',
                'quantity' => 1,
                'unit_price' => $shippingCostsReal,
                'tax_rate' => $taxRateValue,
                'total_amount' => $shippingCostsReal,
                'total_tax_amount' => $shippingCostsTax,
            ];
        }

        // TODO: Handle shipment discount (commented out because its not clear how to handle tax for shipping discount. will be done later)
//        foreach ($order->getAdjustments(AdjustmentInterface::ORDER_SHIPPING_PROMOTION_ADJUSTMENT) as $shipmentDiscount) {
//            $lineItems[] = [
//                'type' => 'discount',
//                'name' => 'Shipping',
//                'quantity' => 1,
//                'unit_price' => $shipmentDiscount->getAmount(),
//                'tax_rate' => 0,
//                'total_amount' => $shipmentDiscount->getAmount(),
//                'total_tax_amount' => 0,
//            ];
//        }

        return $lineItems;
    }

    /**
     * @param OrderInterface $order
     * @param array          $sessionRequestData
     *
     * @return array
     */
    public function generateCountryData(OrderInterface $order, array $sessionRequestData): array
    {
        $address = $order->getBillingAddress();
        $address = $address ?? $order->getShippingAddress();
        $countryCode = $address ? $address->getCountryCode() : 'DE';
        $countryCode = $countryCode ?? 'DE';

        $sessionRequestData['purchase_country'] = $countryCode;
        $sessionRequestData['purchase_currency'] = $order->getCurrencyCode();
        $sessionRequestData['locale'] = $this->generateLocale($order->getLocaleCode());

        return $sessionRequestData;
    }

    /**
     * @param null|AddressInterface $address
     * @param null|CustomerInterface $customer
     *
     * @return array|null
     */
    public function generateAddress(?AddressInterface $address, ?CustomerInterface $customer)
    {
        if (null === $address) {
            return null;
        }

        return [
            'given_name' => $address->getFirstName(),
            'family_name' => $address->getLastName(),
            'email' => null === $customer ? null : $customer->getEmail(),
            'street_address' => $address->getStreet(),
            'postal_code' => $address->getPostcode(),
            'city' => $address->getCity(),
            'region' => $address->getProvinceCode(),
            'phone' => $address->getPhoneNumber(),
            'country' => $address->getCountryCode()
        ];
    }

    /**
     * @param OrderInterface $order
     *
     * @return array
     */
    public function generateShippingInfo(OrderInterface $order)
    {
        $shippingInfo = [];

        foreach ($order->getShipments() as $shipment) {
            $shippingInfo[] = [
                'shipping_company' => $shipment->getMethod()->getName(),
                'shipping_method' => 'Home',
            ];
        }

        return $shippingInfo;
    }

    /**
     * @param string $locale
     *
     * @return mixed|string
     */
    public function generateLocale(string $locale)
    {
        $klarnaLocale = $locale;
        // try to guess a rfc1766 locale if iso639 locale was given
        if (2 === strlen($locale)) {
            $klarnaLocale = strtolower($locale) . '_' . strtoupper($locale);
        }

        // validate locale and use first existing locale that starts with $locale_ when invalid
        if (!Locales::exists($klarnaLocale)) {
            $locales = Locales::getLocales();
            $locales = array_filter($locales, function($localeCode) use ($locale) {
                $len = strlen($locale) + 1;
                return substr($localeCode, 0, $len) === $locale . '_'
                    && !preg_match('/[^A-Za-z_]/', $localeCode);
            });
            // fallback to germany
            $klarnaLocale = array_shift($locales) ?? 'de_DE';
        }

        // klarna expects minus instead of underscore
        $klarnaLocale = str_replace('_', '-', $klarnaLocale);

        return $klarnaLocale;
    }

    /**
     * @param OrderItemInterface $orderItem
     *
     * @return int
     */
    protected function getUnitDiscount(OrderItemInterface $orderItem)
    {
        $discount = 0;
        foreach ($orderItem->getUnits() as $itemUnit) {
            $discount += abs($itemUnit->getAdjustmentsTotal(AdjustmentInterface::ORDER_ITEM_PROMOTION_ADJUSTMENT));
            $discount += abs($itemUnit->getAdjustmentsTotal(AdjustmentInterface::ORDER_PROMOTION_ADJUSTMENT));
            $discount += abs($itemUnit->getAdjustmentsTotal(AdjustmentInterface::ORDER_UNIT_PROMOTION_ADJUSTMENT));
        }

        return intval($discount);
    }
}
