<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Klarna\Action;

use ArrayAccess;
use Exception;
use Lifestyle\Sylius\Klarna\Action\Api\BaseApiAwareAction;
use Payum\Core\Action\ActionInterface;
use Payum\Core\ApiAwareInterface;
use Payum\Core\Bridge\Spl\ArrayObject;
use Payum\Core\Exception\RequestNotSupportedException;
use Payum\Core\GatewayAwareInterface;
use Payum\Core\GatewayAwareTrait;
use Payum\Core\Request\Capture;
use Sylius\Component\Core\Model\OrderInterface;

/**
 * Class PlaceOrderAction
 *
 * @package Lifestyle\Sylius\Klarna\Action\Api
 */
class CaptureAction extends BaseApiAwareAction implements ActionInterface, GatewayAwareInterface, ApiAwareInterface
{
    use GatewayAwareTrait;

    /**
     * @var bool
     */
    private $capture;

    /**
     * CaptureAction constructor.
     *
     * @param bool $capture
     */
    public function __construct(bool $capture)
    {
        parent::__construct();
        $this->capture = $capture;
    }

    /**
     * {@inheritdoc}
     *
     * @param $request Capture
     */
    public function execute($request)
    {
        RequestNotSupportedException::assertSupports($this, $request);
        $details = ArrayObject::ensureArrayObject($request->getModel());

        try {
            if ($this->capture && null !== $details->get('orderId', null)) {
                /** @var OrderInterface $order */
                $orderManagementOrder = $this->api->getApiFactory()->orderManagementOrder($details['orderId']);
                $order = $request->getFirstModel()->getOrder();

                $sessionRequestData = [
                    'captured_amount' => $order->getTotal(),
                ];

                $sessionRequestData['shipping_info'] = $this->api->getKlarnaRequest()->generateShippingInfo($order);
                $sessionRequestData['order_lines'] = $this->api->getKlarnaRequest()->generateOrderLineItems($order);

                $orderManagementOrder->createCapture($sessionRequestData);
                $details['capture'] = true;
            }
        } catch (Exception $exception) {
            $detailMessages = $details->get('messages', []);
            $detailMessages[__CLASS__ . '::' . __FUNCTION__] = $exception->getMessage();
            $details->defaults([
                'messages' => $detailMessages
            ]);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function supports($request)
    {
        return
            $request instanceof Capture
            && $request->getModel() instanceof ArrayAccess
        ;
    }
}
