<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Klarna\Action;

use ArrayAccess;
use Exception;
use Lifestyle\Sylius\Klarna\Action\Api\BaseApiAwareAction;
use Lifestyle\Sylius\Payum\Exception\RefundNotAllowedException;
use Lifestyle\Sylius\Payum\Exception\RefundNotCapturedException;
use Lifestyle\Sylius\Payum\Exception\RefundOrderAlreadyRefundedException;
use Lifestyle\Sylius\Payum\Exception\RefundOrderNotFoundException;
use Payum\Core\Bridge\Spl\ArrayObject;
use Payum\Core\Exception\RequestNotSupportedException;
use Payum\Core\Request\Refund;
use Sylius\Component\Core\Model\OrderInterface;

/**
 * Class RefundAction
 * @package Lifestyle\Sylius\Klarna\Action
 */
class RefundAction extends BaseApiAwareAction
{
    /**
     * {@inheritDoc}
     *
     * @param Refund $request
     *
     * @throws RefundNotAllowedException
     * @throws RefundNotCapturedException
     * @throws RefundOrderNotFoundException
     */
    public function execute($request)
    {
        RequestNotSupportedException::assertSupports($this, $request);
        $details = ArrayObject::ensureArrayObject($request->getModel());

        try {
            /** @var OrderInterface $order */
            $orderManagementOrder = $this->api->getApiFactory()->orderManagementOrder($details['orderId']);
            $orderData = $orderManagementOrder->fetch();

            if ('CAPTURED' !== $orderData['status']) {
                throw new RefundNotCapturedException();
            }
            if (!empty($orderData['refunds'])) {
                throw new RefundOrderAlreadyRefundedException();
            }

            $order = $request->getFirstModel()->getOrder();
            $refundRequestData = [
                'refunded_amount' => $order->getTotal(),
            ];
            $refundRequestData['order_lines'] = $this->api->getKlarnaRequest()->generateOrderLineItems($order);

            $orderManagementOrder->refund($refundRequestData);
        } catch (RefundNotCapturedException $exception) {
            throw $exception;
        } catch (Exception $exception) {
            if (403 === $exception->getCode()) {
                $exception = new RefundNotAllowedException($exception->getMessage());
            } elseif (404 === $exception->getCode()) {
                $exception = new RefundOrderNotFoundException($exception->getMessage());
            }
            throw $exception;
        }
    }

    /**
     * {@inheritDoc}
     */
    public function supports($request)
    {
        return
            $request instanceof Refund
            && $request->getModel() instanceof ArrayAccess
        ;
    }
}
