<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Klarna\Action\Api;

use ArrayAccess;
use Exception;
use Lifestyle\Sylius\Klarna\Request\Api\CreateSession;
use Payum\Core\Action\ActionInterface;
use Payum\Core\ApiAwareInterface;
use Payum\Core\Bridge\Spl\ArrayObject;
use Payum\Core\Exception\RequestNotSupportedException;
use Payum\Core\GatewayAwareInterface;
use Sylius\Component\Core\Model\OrderInterface;

/**
 * Class CreateSessionAction
 *
 * @package Lifestyle\Sylius\Klarna\Action\Api
 */
class CreateSessionAction extends BaseApiAwareAction implements
    ActionInterface,
    GatewayAwareInterface,
    ApiAwareInterface
{
    /**
     * @var ArrayObject
     */
    private $config;

    /**
     * CaptureAction constructor.
     *
     * @param ArrayObject $config
     */
    public function __construct(ArrayObject $config)
    {
        parent::__construct();
        $this->config = $config;
    }

    /**
     * {@inheritdoc}
     *
     * @param $request CreateSession
     */
    public function execute($request)
    {
        RequestNotSupportedException::assertSupports($this, $request);
        $details = ArrayObject::ensureArrayObject($request->getModel());

        try {
            /** @var OrderInterface $order */
            $order = $request->getFirstModel()->getOrder();
            $customer = $order->getCustomer();

            $sessionRequestData = [
                'customer' => [
                    'date_of_birth' => null !== $customer && null !== $customer->getBirthday()
                                       ? $customer->getBirthday()->format('Y-m-d') : null,
                    'gender' => 'm' === $customer->getGender() ? 'male' : 'female',
                ],
                'options' => [
                    'color_border' => $this->config['color_border'],
                    'color_border_selected' => $this->config['color_border_selected'],
                    'color_button' => $this->config['color_button'],
                    'color_button_text' => $this->config['color_button_text'],
                    'color_checkbox' => $this->config['color_checkbox'],
                    'color_checkbox_checkmark' => $this->config['color_checkbox_checkmark'],
                    'color_details' => $this->config['color_details'],
                    'color_header' => $this->config['color_header'],
                    'color_link' => $this->config['color_link'],
                    'color_text' => $this->config['color_text'],
                    'color_text_secondary' => $this->config['color_text_secondary'],
                    'radius_border' => $this->config['radius_border']
                ]
            ];

            $sessionRequestData = $this->api->getKlarnaRequest()->generateCountryData($order, $sessionRequestData);
            $sessionRequestData = $this->api->getKlarnaRequest()->generateOrder($order, $sessionRequestData);

            $sessionRequestData['billing_address'] = $this->api->getKlarnaRequest()
                ->generateAddress($order->getBillingAddress(), $customer);
            $sessionRequestData['shipping_address'] = $this->api->getKlarnaRequest()
                ->generateAddress($order->getShippingAddress(), $customer);
            $sessionRequestData['order_lines'] = $this->api->getKlarnaRequest()->generateOrderLineItems($order);

            $session = $this->api->getApiFactory()->session($details->get('sessionId', null));

            if (null === $details->get('clientToken', null)) {
                $session->create($sessionRequestData);
                $sessionId = $session->getId();
                $details['sessionId'] = $sessionId;
                $details['clientToken'] = $session['client_token'];
                $details['paymentMethods'] = $session['payment_method_categories'] ?? [];
            } else {
                $session->update($sessionRequestData);
                // read the session after updating to get payment method changes
                $session->fetch();
                $details['paymentMethods'] = $session['payment_method_categories'] ?? [];
            }
        } catch (Exception $exception) {
            $detailMessages = $details->get('messages', []);
            $detailMessages[__CLASS__ . '::' . __FUNCTION__] = $exception->getMessage();
            $details->defaults([
                'messages' => $detailMessages
            ]);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function supports($request)
    {
        return
            $request instanceof CreateSession
            && $request->getModel() instanceof ArrayAccess
        ;
    }
}
