<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Klarna\Action\Api;

use ArrayAccess;
use Exception;
use Lifestyle\Sylius\Klarna\Request\Api\PlaceOrder;
use Payum\Core\Action\ActionInterface;
use Payum\Core\ApiAwareInterface;
use Payum\Core\Bridge\Spl\ArrayObject;
use Payum\Core\Exception\RequestNotSupportedException;
use Payum\Core\GatewayAwareInterface;
use Payum\Core\Reply\HttpRedirect;
use Sylius\Component\Core\Model\OrderInterface;

/**
 * Class PlaceOrderAction
 *
 * @package Lifestyle\Sylius\Klarna\Action\Api
 */
class PlaceOrderAction extends BaseApiAwareAction implements ActionInterface, GatewayAwareInterface, ApiAwareInterface
{
    /**
     * {@inheritdoc}
     *
     * @param $request PlaceOrder
     */
    public function execute($request)
    {
        RequestNotSupportedException::assertSupports($this, $request);
        $details = ArrayObject::ensureArrayObject($request->getModel());

        try {
            // back to authorize
            if (null === $details->get('authToken', null) || empty($details->get('authToken', null))) {
                throw new HttpRedirect($request->getToken()->getTargetUrl());
            }
            // order was already placed >> forward to after url
            if (null !== $details->get('orderId', null)) {
                throw new HttpRedirect($request->getToken()->getAfterUrl());
            }

            /** @var OrderInterface $order */
            $order = $request->getFirstModel()->getOrder();
            $customer = $order->getCustomer();

            $sessionRequestData = [
                'merchant_urls' => [
                    'confirmation' => $request->getToken()->getAfterUrl()
                ]
            ];

            $sessionRequestData = $this->api->getKlarnaRequest()->generateCountryData($order, $sessionRequestData);
            $sessionRequestData = $this->api->getKlarnaRequest()->generateOrder($order, $sessionRequestData);

            $sessionRequestData['billing_address'] = $this->api->getKlarnaRequest()
                ->generateAddress($order->getBillingAddress(), $customer);
            $sessionRequestData['shipping_address'] = $this->api->getKlarnaRequest()
                ->generateAddress($order->getShippingAddress(), $customer);
            $sessionRequestData['order_lines'] = $this->api->getKlarnaRequest()->generateOrderLineItems($order);

            $order = $this->api->getApiFactory()->order($details['authToken']);
            $response = $order->create($sessionRequestData);

            if (!isset($response['fraud_status']) || 'ACCEPTED' !== $response['fraud_status']) {
                throw new HttpRedirect($request->getToken()->getTargetUrl());
            }

            $details['orderId'] = $response['order_id'];
            throw new HttpRedirect($response['redirect_url']);
        } catch (HttpRedirect $redirect) {
            throw $redirect;
        } catch (Exception $exception) {
            $detailMessages = $details->get('messages', []);
            $detailMessages[__CLASS__ . '::' . __FUNCTION__] = $exception->getMessage();
            $details->defaults([
                'messages' => $detailMessages
            ]);
            throw new HttpRedirect($request->getToken()->getTargetUrl());
        }
    }

    /**
     * {@inheritdoc}
     */
    public function supports($request)
    {
        return
            $request instanceof PlaceOrder
            && $request->getModel() instanceof ArrayAccess
        ;
    }
}
