<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author    Franz Weisflug
 * @copyright 2019 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Lifestyle\Sylius\Location\Consumer\Helper;

use Doctrine\ORM\EntityManager;
use Lifestyle\Sylius\Account\Consumer\Services\Shared\Helper\DatabaseTrait;
use Lifestyle\Sylius\Location\Consumer\Asset\VersionStrategy\JsonManifestVersionStrategy;
use Lifestyle\Sylius\Location\Consumer\Configuration\LocationConfiguration;
use Lifestyle\Sylius\Location\Consumer\Entity\Location;
use Psr\Log\LoggerInterface;
use Sylius\Bundle\ResourceBundle\Doctrine\ORM\EntityRepository;
use Sylius\Component\Resource\Factory\FactoryInterface as ResourceFactory;
use Symfony\Component\Asset\Packages;

/**
 * Class DocumentHelper
 * @package Lifestyle\Sylius\Location\Consumer\Helper
 */
class DocumentHelper
{
    use DatabaseTrait;

    /**
     * @var JsonManifestVersionStrategy
     */
    private $assetVersionStrategy;

    /**
     * @var LocationConfiguration
     */
    private $locationConfiguration;

    /**
     * @var EntityRepository
     */
    private $locationRepository;

    /**
     * @var EntityManager
     */
    private $locationManager;

    /**
     * @var Packages
     */
    private $packages;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * DocumentHelper constructor.
     * @param JsonManifestVersionStrategy $assetVersionStrategy
     * @param LocationConfiguration $locationConfiguration
     * @param EntityRepository $locationRepository
     * @param EntityManager $locationManager
     * @param Packages $packages
     * @param LoggerInterface $logger
     */
    public function __construct(
        JsonManifestVersionStrategy $assetVersionStrategy,
        LocationConfiguration $locationConfiguration,
        EntityRepository $locationRepository,
        EntityManager $locationManager,
        Packages $packages,
        LoggerInterface $logger
    ) {
        $this->assetVersionStrategy = $assetVersionStrategy;
        $this->locationConfiguration = $locationConfiguration;
        $this->locationRepository = $locationRepository;
        $this->locationManager = $locationManager;
        $this->packages = $packages;
        $this->logger = $logger;
    }

    /**
     * @param int $resourceId
     */
    public function updateDocument(int $resourceId)
    {
        // reconnect database if connection was lost
        $this->locationManager = $this->getClearedEntityManager($this->locationManager);

        $this->assetVersionStrategy->clearCache();
        $url = $this->packages->getUrl($resourceId, $this->locationConfiguration->getPackageName());
        if ('/' . $resourceId === $url) {
            $url = null;
        }

        /** @var Location $location */
        $locations = $this->locationRepository->findBy(['benchPlanId' => $resourceId]);
        foreach ($locations as $location) {
            $location->setBenchPlan($url);
            $this->locationRepository->add($location);
            $this->logger->info('Bench Plan (' . $resourceId .  ') in Location (' . $location->getCode() . ') added');
        }
    }

    /**
     * @param int $resourceId
     */
    public function removeDocument(int $resourceId)
    {
        // reconnect database if connection was lost
        $this->locationManager = $this->getClearedEntityManager($this->locationManager);

        /** @var Location $location */
        $locations = $this->locationRepository->findBy(['benchPlanId' => $resourceId]);
        foreach ($locations as $location) {
            $location->setBenchPlanId(null);
            $location->setBenchPlan(null);
            $this->locationRepository->add($location);
            $this->logger->info('Bench Plan (' . $resourceId .  ') from Location (' . $location->getCode() . ') removed');
        }
    }
}
