<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      Franz Weisflug
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Location\Consumer\Services\Location\Version1;

use Doctrine\ORM\EntityManager;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\Account\Consumer\Entity\Customer;
use Lifestyle\Sylius\Account\Consumer\Entity\Location as AccountLocation;
use Lifestyle\Sylius\Location\Consumer\Asset\VersionStrategy\JsonManifestVersionStrategy;
use Lifestyle\Sylius\Location\Consumer\Configuration\LocationConfiguration;
use Lifestyle\Sylius\Location\Consumer\Entity\Location;
use Lifestyle\Sylius\Account\Consumer\Services\Location\Version1\LocationServiceHandler as LocationServiceHandlerAccount;
use Psr\Log\LoggerInterface;
use Sylius\Bundle\CoreBundle\Doctrine\ORM\AddressRepository;
use Sylius\Bundle\CoreBundle\Doctrine\ORM\CustomerRepository;
use Sylius\Bundle\ResourceBundle\Doctrine\ORM\EntityRepository;
use Sylius\Component\Resource\Factory\FactoryInterface as ResourceFactory;
use Symfony\Component\Asset\Packages;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Class LocationServiceHandler
 *
 * @package Lifestyle\Sylius\Location\Consumer\Services\Location\Version1
 */
class LocationServiceHandler extends LocationServiceHandlerAccount
{
    /**
     * @var CustomerRepository
     */
    private $customerRepository;

    /**
     * @var JsonManifestVersionStrategy
     */
    private $assetVersionStrategy;

    /**
     * @var LocationConfiguration
     */
    private $locationConfiguration;

    /**
     * @var Packages
     */
    private $packages;

    /**
     * LocationServiceHandler constructor.
     *
     * @param ResourceFactory             $locationFactory
     * @param EntityRepository            $locationRepository
     * @param EntityManager               $locationManager
     * @param AddressRepository           $addressRepository
     * @param ValidatorInterface          $validator
     * @param CustomerRepository          $customerRepository
     * @param JsonManifestVersionStrategy $assetVersionStrategy
     * @param LocationConfiguration       $locationConfiguration
     * @param Packages                    $packages
     * @param LoggerInterface             $logger
     */
    public function __construct(
        ResourceFactory $locationFactory,
        EntityRepository $locationRepository,
        EntityManager $locationManager,
        AddressRepository $addressRepository,
        ValidatorInterface $validator,
        CustomerRepository $customerRepository,
        JsonManifestVersionStrategy $assetVersionStrategy,
        LocationConfiguration $locationConfiguration,
        Packages $packages,
        LoggerInterface $logger
    ) {
        parent::__construct(
            $locationFactory,
            $locationRepository,
            $locationManager,
            $addressRepository,
            $validator,
            $logger
        );

        $this->customerRepository = $customerRepository;
        $this->assetVersionStrategy = $assetVersionStrategy;
        $this->locationConfiguration = $locationConfiguration;
        $this->packages = $packages;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param AccountLocation       $location
     */
    protected function handleAdditionalData(DataCollectorInterface $collector, AccountLocation $location)
    {
        if (!$location instanceof Location) {
            return;
        }

        $benchPlanId = $collector->getItemValue('BenchPlanId');
        $benchPlanId = empty($benchPlanId) ? null : $benchPlanId;

        $location->setRestaurantNumber($collector->getItemValue('restaurantNr'));
        $location->setConcept($collector->getItemValue('concept'));
        $location->setBenchPlanId($benchPlanId);

        $this->assetVersionStrategy->clearCache();
        $url = $this->packages->getUrl($benchPlanId, $this->locationConfiguration->getPackageName());
        if ('/' . $benchPlanId === $url) {
            $url = null;
        }
        $location->setBenchPlan($url);

        /** @var Customer $customer */
        $customer = $this->customerRepository->findOneBy(['code' => $collector->getItemValue('FranchiseeId')]);
        if (null !== $customer) {
            $location->setCustomer($customer);
        }
    }
}
