<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Location\Consumer\Asset\VersionStrategy;

use Symfony\Component\Asset\VersionStrategy\VersionStrategyInterface;

/**
 * Class JsonManifestVersionStrategy
 *
 * This is nearly a copy of Symfonys
 * @see \Symfony\Component\Asset\VersionStrategy\JsonManifestVersionStrategy
 * We only added the clearCache method to clean up in rabbitmq-consumer-process
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Lifestyle\Sylius\Location\Consumer\Asset\VersionStrategy
 */
class JsonManifestVersionStrategy implements VersionStrategyInterface
{
    /**
     * @var string
     */
    private $manifestPath;

    /**
     * @var array
     */
    private $manifestData;

    /**
     * @param string $manifestPath Absolute path to the manifest file
     */
    public function __construct(string $manifestPath)
    {
        $this->manifestPath = $manifestPath;
    }

    /**
     * With a manifest, we don't really know or care about what
     * the version is. Instead, this returns the path to the
     * versioned file.
     * @param string $path
     * @return string
     */
    public function getVersion($path)
    {
        return $this->applyVersion($path);
    }

    /**
     * @param string $path
     * @return string
     */
    public function applyVersion($path)
    {
        return $this->getManifestPath($path) ?: $path;
    }

    /**
     * Force reload of manifest.json file
     */
    public function clearCache()
    {
        $this->manifestData = null;
    }

    /**
     * @param string$path
     * @return string|null
     */
    private function getManifestPath($path)
    {
        if (null === $this->manifestData) {
            if (!file_exists($this->manifestPath)) {
                throw new \RuntimeException(sprintf('Asset manifest file "%s" does not exist.', $this->manifestPath));
            }

            $this->manifestData = json_decode(file_get_contents($this->manifestPath), true);
            if (0 < json_last_error()) {
                throw new \RuntimeException(sprintf('Error parsing JSON from asset manifest file "%s" - %s', $this->manifestPath, json_last_error_msg()));
            }
        }

        return isset($this->manifestData[$path]) ? $this->manifestData[$path] : null;
    }
}
