<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\OrderInventory\Operator;

use Lifestyle\Sylius\OrderInventory\RequestData\Factory;
use Lifestyle\RabbitMq\Producer\Producer\ProducerInterface;
use Sylius\Component\Core\Inventory\Operator\OrderInventoryOperatorInterface;
use Sylius\Component\Core\Model\ProductVariantInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\OrderItemInterface;
use Sylius\Component\Core\OrderPaymentStates;
use Webmozart\Assert\Assert;

/**
 * Class OrderInventoryOperator
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\OrderInventory\Operator
 */
final class OrderInventoryOperator implements OrderInventoryOperatorInterface
{
    const INVENTORY_GIVE_BACK_EVENT = 'product.inventory.give_back';
    const INVENTORY_RELEASE_EVENT = 'product.inventory.release';
    const INVENTORY_SELL_EVENT = 'product.inventory.sell';
    const INVENTORY_HOLD_EVENT = 'product.inventory.hold';

    /**
     * @var ProducerInterface
     */
    private $producer;

    /**
     * @var Factory
     */
    private $requestDataFactory;

    /**
     * OrderInventoryOperator constructor.
     * @param ProducerInterface $producer
     * @param Factory $requestDataFactory
     */
    public function __construct(ProducerInterface $producer, Factory $requestDataFactory)
    {
        $this->producer = $producer;
        $this->requestDataFactory = $requestDataFactory;
    }

    /**
     * {@inheritdoc}
     */
    public function cancel(OrderInterface $order): void
    {
        if (in_array(
            $order->getPaymentState(),
            [OrderPaymentStates::STATE_PAID, OrderPaymentStates::STATE_REFUNDED],
            true
        )) {
            $this->giveBack($order);

            return;
        }

        $this->release($order);
    }

    /**
     * @param OrderInterface $order
     */
    private function giveBack(OrderInterface $order): void
    {
        /** @var OrderItemInterface $orderItem */
        foreach ($order->getItems() as $orderItem) {
            $variant = $orderItem->getVariant();

            if (!$variant->isTracked()) {
                continue;
            }

            $variant->setOnHand($variant->getOnHand() + $orderItem->getQuantity());

            $this->send($variant, $orderItem, self::INVENTORY_GIVE_BACK_EVENT);
        }
    }

    /**
     * @param OrderInterface $order
     * @throws \InvalidArgumentException
     */
    private function release(OrderInterface $order): void
    {
        /** @var OrderItemInterface $orderItem */
        foreach ($order->getItems() as $orderItem) {
            $variant = $orderItem->getVariant();

            if (!$variant->isTracked()) {
                continue;
            }

            Assert::greaterThanEq(
                ($variant->getOnHold() - $orderItem->getQuantity()),
                0,
                sprintf(
                    'Not enough units to decrease on hold quantity from the inventory of a variant "%s".',
                    $variant->getName()
                )
            );

            $variant->setOnHold($variant->getOnHold() - $orderItem->getQuantity());

            $this->send($variant, $orderItem, self::INVENTORY_RELEASE_EVENT);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function hold(OrderInterface $order): void
    {
        /** @var OrderItemInterface $orderItem */
        foreach ($order->getItems() as $orderItem) {
            $variant = $orderItem->getVariant();

            if (!$variant->isTracked()) {
                continue;
            }

            $variant->setOnHold($variant->getOnHold() + $orderItem->getQuantity());

            $this->send($variant, $orderItem, self::INVENTORY_HOLD_EVENT);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function sell(OrderInterface $order): void
    {
        /** @var OrderItemInterface $orderItem */
        foreach ($order->getItems() as $orderItem) {
            $variant = $orderItem->getVariant();

            if (!$variant->isTracked()) {
                continue;
            }

            Assert::greaterThanEq(
                ($variant->getOnHold() - $orderItem->getQuantity()),
                0,
                sprintf(
                    'Not enough units to decrease on hold quantity from the inventory of a variant "%s".',
                    $variant->getName()
                )
            );

            Assert::greaterThanEq(
                ($variant->getOnHand() - $orderItem->getQuantity()),
                0,
                sprintf(
                    'Not enough units to decrease on hand quantity from the inventory of a variant "%s".',
                    $variant->getName()
                )
            );

            $variant->setOnHold($variant->getOnHold() - $orderItem->getQuantity());
            $variant->setOnHand($variant->getOnHand() - $orderItem->getQuantity());

            $this->send($variant, $orderItem, self::INVENTORY_SELL_EVENT);
        }
    }

    /**
     * @param ProductVariantInterface $variant
     * @param OrderItemInterface $orderItem
     * @param string $event
     */
    private function send(ProductVariantInterface $variant, OrderItemInterface $orderItem, string $event): void
    {
        $dataCollector = $this->requestDataFactory->dataCollector();
        $dataCollector->addItem('id', $orderItem->getVariant()->getCode());
        $dataCollector->addItem('changedQuantity', $orderItem->getQuantity());
        $dataCollector->addItem('onHold', $variant->getOnHold());

        $this->producer->send($event, $dataCollector);
    }
}
