<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Payum\Controller;

use FOS\RestBundle\View\View;
use Payum\Core\Payum;
use Payum\Core\Request\Generic;
use Sylius\Bundle\PayumBundle\Factory\GetStatusFactoryInterface;
use Sylius\Bundle\PayumBundle\Factory\ResolveNextRouteFactoryInterface;
use Sylius\Bundle\ResourceBundle\Controller\RequestConfigurationFactoryInterface;
use Sylius\Bundle\ResourceBundle\Controller\ViewHandlerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\PaymentInterface;
use Sylius\Component\Order\Repository\OrderRepositoryInterface;
use Sylius\Component\Resource\Metadata\MetadataInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Routing\RouterInterface;

/**
 * Class PayumController
 *
 * @package Lifestyle\Sylius\Payum\Controller
 */
class PayumController
{
    /** @var Payum */
    private $payum;

    /** @var OrderRepositoryInterface */
    private $orderRepository;

    /** @var MetadataInterface */
    private $orderMetadata;

    /** @var RequestConfigurationFactoryInterface */
    private $requestConfigurationFactory;

    /** @var ViewHandlerInterface */
    private $viewHandler;

    /** @var RouterInterface */
    private $router;

    /** @var GetStatusFactoryInterface */
    private $getStatusRequestFactory;

    /** @var ResolveNextRouteFactoryInterface */
    private $resolveNextRouteRequestFactory;

    /**
     * PayumController constructor.
     *
     * @param Payum                                $payum
     * @param OrderRepositoryInterface             $orderRepository
     * @param MetadataInterface                    $orderMetadata
     * @param RequestConfigurationFactoryInterface $requestConfigurationFactory
     * @param ViewHandlerInterface                 $viewHandler
     * @param RouterInterface                      $router
     * @param GetStatusFactoryInterface            $getStatusFactory
     * @param ResolveNextRouteFactoryInterface     $resolveNextRouteFactory
     */
    public function __construct(
        Payum $payum,
        OrderRepositoryInterface $orderRepository,
        MetadataInterface $orderMetadata,
        RequestConfigurationFactoryInterface $requestConfigurationFactory,
        ViewHandlerInterface $viewHandler,
        RouterInterface $router,
        GetStatusFactoryInterface $getStatusFactory,
        ResolveNextRouteFactoryInterface $resolveNextRouteFactory
    ) {
        $this->payum = $payum;
        $this->orderRepository = $orderRepository;
        $this->orderMetadata = $orderMetadata;
        $this->requestConfigurationFactory = $requestConfigurationFactory;
        $this->viewHandler = $viewHandler;
        $this->router = $router;
        $this->getStatusRequestFactory = $getStatusFactory;
        $this->resolveNextRouteRequestFactory = $resolveNextRouteFactory;
    }

    /**
     * @param Request $request
     * @param int     $cartId
     *
     * @return Response
     */
    public function prepareAuthorizeAction(Request $request, int $cartId): Response
    {
        $configuration = $this->requestConfigurationFactory->create($this->orderMetadata, $request);

        /** @var OrderInterface $order */
        $order = $this->orderRepository->findCartById($cartId);

        if (null === $order) {
            throw new NotFoundHttpException(sprintf('Order with token "%s" does not exist.', $cartId));
        }

        $request->getSession()->set('sylius_order_id', $order->getId());
        $payment = $order->getLastPayment(PaymentInterface::STATE_CART);

        if (null === $payment) {
            $url = $this->router->generate('sylius_shop_checkout_select_payment');

            return new RedirectResponse($url);
        }

        // check if use_authorize_before is activated in gateway config
        if (!isset($payment->getMethod()->getGatewayConfig()->getConfig()['authorize_mode'])
            || 'use_authorize_before' !== $payment->getMethod()->getGatewayConfig()->getConfig()['authorize_mode']
        ) {
            return new RedirectResponse(
                $this->router->generate('sylius_shop_checkout_complete')
            );
        }

        $token = $this->payum->getTokenFactory()->createAuthorizeToken(
            $payment->getMethod()->getGatewayConfig()->getGatewayName(),
            $payment,
            $configuration->getParameters()->get('redirect')['route'] ?? null,
            $configuration->getParameters()->get('redirect')['parameters'] ?? []
        );

        return $this->viewHandler->handle(
            $configuration,
            View::createRedirect($token->getTargetUrl())
        );
    }
}
