<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\Payum\EventListener;

use Doctrine\ORM\EntityManager;
use Lifestyle\Sylius\Payum\Exception\CancelNotPayedException;
use Lifestyle\Sylius\Payum\Exception\NoGatewayConfigException;
use Lifestyle\Sylius\Payum\Exception\RefundOrderAlreadyRefundedException;
use Lifestyle\Sylius\Payum\Exception\RefundOrderNotFoundException;
use Lifestyle\Sylius\Payum\Exception\RefundNotAllowedException;
use Lifestyle\Sylius\Payum\Exception\RefundNotCapturedException;
use Payum\Bundle\PayumBundle\Controller\PayumController;
use Payum\Core\GatewayAwareTrait;
use Payum\Core\Model\GatewayConfig;
use Payum\Core\Payum;
use Payum\Core\Request\Cancel;
use Payum\Core\Request\Refund;
use SM\Event\TransitionEvent;
use SM\Factory\FactoryInterface;
use Sylius\Component\Core\Model\Payment;
use Sylius\Component\Core\OrderPaymentTransitions;
use Sylius\Component\Order\Model\OrderInterface;
use Sylius\Component\Payment\PaymentTransitions;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

/**
 * Class OrderListener
 * @package Lifestyle\Sylius\Payum\EventListener
 */
class OrderListener extends PayumController
{
    use GatewayAwareTrait;

    /** @var Payum */
    private $payum;

    /**
     * @var SessionInterface
     */
    private $session;

    /**
     * @var FactoryInterface
     */
    private $stateMachineFactory;

    /**
     * @var EntityManager
     */
    private $cartManager;

    /**
     * OrderListener constructor.
     * @param Payum $payum
     * @param SessionInterface $session
     */
    public function __construct(
        Payum $payum,
        SessionInterface $session,
        FactoryInterface $stateMachineFactory,
        EntityManager $cartManager
    ) {
        $this->payum = $payum;
        $this->session = $session;
        $this->stateMachineFactory = $stateMachineFactory;
        $this->cartManager = $cartManager;
    }

    /**
     * @param TransitionEvent $event
     */
    public function onStateMachinePreTransition(TransitionEvent $event)
    {
        $config = $event->getConfig();
        $stateMachine = $event->getStateMachine();
        $graph = $stateMachine->getGraph();
        $fromState = $event->getState();
        $toState = $config['to'];
        $transition = $event->getTransition();
        $object = $stateMachine->getObject();

        if ('sylius_order' === $graph) {
            if ('new' === $fromState && 'cancelled' === $toState && 'cancel' === $transition) {
                /** @var OrderInterface $object */
                $this->executePayumRequest($event, $object->getPayments()[0], Cancel::class);
            }
        } elseif ('sylius_payment' === $graph) {
            if ('completed' === $fromState && 'refunded' === $toState && 'refund' === $transition) {
                /** @var Payment $object */
                $this->executePayumRequest($event, $object, Refund::class);
            }
        }
    }

    /**
     * @param TransitionEvent $event
     * @param Payment $payment
     * @param string $payumRequestClass
     */
    private function executePayumRequest(
        TransitionEvent $event,
        Payment $payment,
        string $payumRequestClass
    ) {
        $token = null;

        try {
            /** @var GatewayConfig $gatewayConfig */
            $gatewayConfig = $payment->getMethod()->getGatewayConfig();

            if (null === $gatewayConfig) {
                throw new NoGatewayConfigException();
            }

            // create a token with payment identity model
            $identityModel = $this->payum->getStorage(Payment::class)->identify($payment);
            $token = $this->payum->getTokenFactory()
                ->createRefundToken($gatewayConfig->getGatewayName(), $identityModel);

            $gateway = $this->payum->getGateway($token->getGatewayName());
            $gateway->execute(new $payumRequestClass($token));

            $this->updatePaymentStateRefunded($event, $payumRequestClass);
        } catch (RefundNotCapturedException $exception) {
            $this->session->getBag('flashes')->add('warning', 'Order was not captured, no refund Required.');
            $event->setRejected(true);
        } catch (RefundNotAllowedException $exception) {
            $this->session->getBag('flashes')->add(
                'warning',
                'Could not refund payment, please check Order in Klarna Admin: ' . $exception->getMessage()
            );
            $event->setRejected(true);
        } catch (RefundOrderNotFoundException $exception) {
            $this->session->getBag('flashes')->add(
                'warning',
                'Could not find Order, please check Order in Klarna Admin: ' . $exception->getMessage()
            );
            $event->setRejected(true);
        } catch (RefundOrderAlreadyRefundedException $exception) {
            $this->session->getBag('flashes')->add('info', 'Order already refunded, no refund required.');
            $this->updatePaymentStateRefunded($event, $payumRequestClass);
        } catch (CancelNotPayedException $exception) {
            $this->session->getBag('flashes')->add('info', 'Order was not payed yet, no refund required.');
        } catch (NoGatewayConfigException $exception) {
            // No output
        } catch (\Exception $exception) {
            $this->session->getBag('flashes')->add('error', $exception->getMessage());
            $event->setRejected(true);
        }

        if (null !== $token) {
            $this->payum->getHttpRequestVerifier()->invalidate($token);
        }
    }

    /**
     * @param TransitionEvent $event
     * @param string $payumRequestClass
     */
    private function updatePaymentStateRefunded(TransitionEvent $event, string $payumRequestClass)
    {
        $object = $event->getStateMachine()->getObject();

        if ($object instanceof OrderInterface && Cancel::class === $payumRequestClass) {
            /** @var Payment $payment */
            $payment = $object->getPayments()[0];

            if ('completed' === $payment->getState()) {
                try {
                    $orderStateMachine = $this->stateMachineFactory->get($object, OrderPaymentTransitions::GRAPH);
                    $orderStateMachine->apply(OrderPaymentTransitions::TRANSITION_REFUND);

                    $paymentStateMachine = $this->stateMachineFactory->get($payment, PaymentTransitions::GRAPH);
                    $paymentStateMachine->apply(PaymentTransitions::TRANSITION_REFUND);

                    $this->cartManager->flush();
                } catch (\Exception $exception) {
                    $this->session->getBag('flashes')->add(
                        'warning',
                        'Could not update Payment state' . $exception->getMessage()
                    );
                }
            }
        }
    }
}
