<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace LifeStyle\Sylius\PricingPlugin\Factory;

use LifeStyle\Sylius\PricingPlugin\Entity\CustomDiscountInterface;
use LifeStyle\Sylius\PricingPlugin\Entity\ProductVariantInterface;
use LifeStyle\Sylius\PricingPlugin\Entity\TierPrice;
use LifeStyle\Sylius\PricingPlugin\Entity\TierPriceInterface;
use Doctrine\ORM\EntityNotFoundException;
use Sylius\Plugin\CorePlugin\Fixture\OptionsResolver\LazyOption;
use Sylius\Component\Channel\Repository\ChannelRepositoryInterface;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Repository\ProductVariantRepositoryInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

/**
 * Class CustomDiscountsFactory
 * @package LifeStyle\Sylius\PricingPlugin\Factory
 */
class CustomDiscountsFactory implements CustomDiscountsFactoryInterface
{
    /**
     * @var ProductVariantRepositoryInterface
     */
    private $productVariantRepository;

    /**
     * @var ChannelRepositoryInterface
     */
    private $channelRepository;

    /**
     * TierPriceFactory constructor.
     * @param ProductVariantRepositoryInterface $productVariantRepository
     * @param ChannelRepositoryInterface $channelRepository
     */
    public function __construct(
        ProductVariantRepositoryInterface $productVariantRepository,
        ChannelRepositoryInterface $channelRepository
    ) {
        $this->productVariantRepository = $productVariantRepository;
        $this->channelRepository        = $channelRepository;
    }

    /**
     * @param array $options
     * @return TierPriceInterface
     * @throws EntityNotFoundException
     */
    public function create(array $options = []): TierPriceInterface
    {
        /** @var ProductVariantInterface|null $productVariant */
        $productVariant = $this->productVariantRepository->findOneBy(['code' => $options['product_variant']]);

        if ($productVariant === null) {
            throw new EntityNotFoundException('Create the product variant first');
        }
        dump($options);
        return $this->createAtProductVariant($productVariant, $options);
    }

    /**
     * @param OptionsResolver $resolver
     */
    protected function configureOption(OptionsResolver $resolver): void
    {
        $resolver->setDefault('quantity', 1);
        $resolver->setAllowedTypes('quantity', 'integer');

        $resolver->setDefault('price', 0);
        $resolver->setAllowedTypes('price', 'integer');

        $resolver->setDefault('product_variant', LazyOption::randomOne($this->productVariantRepository));
        $resolver->setAllowedTypes('product_variant', ProductVariantInterface::class);

        $resolver->setDefault('channel', LazyOption::randomOne($this->channelRepository));
        $resolver->setAllowedTypes('channel', ChannelInterface::class);
    }

    /**
     * @param ProductVariantInterface $productVariant
     * @param array $options
     * @return TierPriceInterface
     */
    public function createAtProductVariant(
        ProductVariantInterface $productVariant,
        array $options = []
    ): CustomDiscountInterface {
        $tierPrice = new TierPrice();

        $tierPrice->setQty($options['quantity']);
        $tierPrice->setProductVariant($productVariant);

        $tierPrice->setChannel($this->channelRepository->findOneBy(['code' => $options['channel']]));
        $tierPrice->setPrice($options['price']);

        $productVariant->addTierPrice($tierPrice);

        return $tierPrice;
    }
}
