<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace LifeStyle\Sylius\PricingPlugin\Traits;

use LifeStyle\Sylius\PricingPlugin\Entity\ProductVariant;
use LifeStyle\Sylius\PricingPlugin\Entity\CustomDiscount;
use Doctrine\Common\Collections\ArrayCollection;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Model\ProductVariantInterface;

/**
 * Trait CustomDiscountableTrait
 * @package LifeStyle\Sylius\PricingPlugin\Traits
 */
trait CustomDiscountableTrait
{
    /**
     * @var ArrayCollection
     */
    protected $customDiscounts;

    /**
     * @return void
     */
    public function initCustomDiscountableTrait(): void
    {
        $this->customDiscounts = new ArrayCollection();
    }

    /**
     * @return CustomDiscountableInterface[]
     */
    public function getCustomDiscounts(): array
    {
        if (null === $this->customDiscounts) {
            $this->initCustomDiscountableTrait();
        }

        return $this->customDiscounts->toArray();
    }

    /**
     * @param ChannelInterface $channel
     * @return CustomDiscountableInterface[]
     */
    public function getCustomDiscountsForChannel(ChannelInterface $channel): array
    {
        return array_filter($this->getCustomDiscounts(), function (CustomDiscount $customDiscount) use ($channel) {
            $customDiscountChannel = $customDiscount->getChannel();

            return $customDiscountChannel === null ? false : $customDiscountChannel->getId() === $channel->getId();
        });
    }

    /**
     * @param ChannelInterface $channel
     * @param int|null $priceListId
     * @return array
     */
    public function getCustomDiscountsForChannelAndPriceList(ChannelInterface $channel, int $priceListId = null): array
    {
        if (null === $priceListId) {
            return [];
        }

        return array_filter($this->getCustomDiscounts(),
            function (CustomDiscount $customDiscount) use ($channel, $priceListId) {
                $customDiscountChannel = $customDiscount->getChannel();
                return
                    $customDiscountChannel === null ?
                        false : $customDiscountChannel->getId() === $channel->getId()
                        && $customDiscount->getPriceList()->getId() === $priceListId;
            });
    }

    /**
     * @param CustomDiscount $customDiscount
     */
    public function removeCustomDiscount(CustomDiscount $customDiscount): void
    {
        $this->customDiscounts->removeElement($customDiscount);
    }

    /**
     * @param CustomDiscount $customDiscount
     */
    public function addCustomDiscount(CustomDiscount $customDiscount): void
    {
        $customDiscount->setProductVariant($this);
        $this->customDiscounts->add($customDiscount);
    }

    /**
     * @param array $customDiscounts
     */
    public function setCustomDiscounts(array $customDiscounts): void
    {
        if (!$this instanceof ProductVariantInterface) {
            return;
        }

        $this->customDiscounts = new ArrayCollection();

        foreach ($customDiscounts as $customDiscount) {
            /** @var CustomDiscount $customDiscount */
            $this->addCustomDiscount($customDiscount);
        }
    }
}
