<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace LifeStyle\Sylius\PricingPlugin\Services;

use LifeStyle\Sylius\PricingPlugin\Traits\CustomDiscountableInterface;
use LifeStyle\Sylius\PricingPlugin\Traits\TierPriceableInterface;
use Sylius\Component\Core\Calculator\ProductVariantPriceCalculatorInterface;
use Sylius\Component\Core\Model\ProductVariantInterface;
use Sylius\Component\Core\Model\ShopUser;
use Sylius\Component\Core\Model\ShopUserInterface;

/**
 * Class ProductVariantPriceCalculator
 * @package LifeStyle\Sylius\PricingPlugin\Services
 */
final class ProductVariantPriceCalculator implements ProductVariantPriceCalculatorInterface
{
    /**
     * @var ProductVariantPriceCalculatorInterface
     */
    private $basePriceCalculator;

    /**
     * @var TierPriceFinderInterface
     */
    private $tierPriceFinder;

    /**
     * @var CustomDiscountFinderInterface
     */
    private $customDiscountFinder;

    /**
     * @var ShopUserManagerInterface
     */
    private $shopUserManager;

    /**
     * @var ShopUserInterface|null
     */
    private $shopUser = null;

    /**
     * ProductVariantPriceCalculator constructor.
     * @param ProductVariantPriceCalculatorInterface $basePriceCalculator
     * @param TierPriceFinderInterface $tierPriceFinder
     * @param CustomDiscountFinderInterface $customDiscountFinder
     * @param ShopUserManagerInterface $shopUserManager
     */
    public function __construct(
        ProductVariantPriceCalculatorInterface $basePriceCalculator,
        TierPriceFinderInterface $tierPriceFinder,
        CustomDiscountFinderInterface $customDiscountFinder,
        ShopUserManagerInterface $shopUserManager
    ) {
        $this->basePriceCalculator = $basePriceCalculator;
        $this->tierPriceFinder = $tierPriceFinder;
        $this->customDiscountFinder = $customDiscountFinder;
        $this->shopUserManager = $shopUserManager;
    }


    /**
     * @param ProductVariantInterface $productVariant
     * @param array $context
     * @return int
     */
    public function calculate(ProductVariantInterface $productVariant, array $context): int
    {
        //check shopUser here and get his priceList from group
        if(null === $this->shopUser){
            $shopUser = $this->shopUserManager->getShopUser();
        } else {
            $shopUser = $this->shopUser;
        }

        $unmodifiedPrice = $this->basePriceCalculator->calculate($productVariant, $context);

        //only shopUser with a group and assigned price list can get tier prices or discounts
        if (null !== $shopUser && null !== $shopUser->getCustomer()->getGroup()) {

            $shopUserPriceLists = $this->getPriceListsForUser($shopUser);

            //if the users group has no pricelist don't recalculate the price
            if (empty($shopUserPriceLists)) {
                return $unmodifiedPrice;
            }

            //first check for quantity and tier prices
            $tierReducedPrice = null;
            if (array_key_exists('quantity', $context)) {
                if ($productVariant instanceof TierPriceableInterface) {
                    $tierPrice = $this->tierPriceFinder->find(
                        $productVariant,
                        $context['channel'],
                        $context['quantity'],
                        $shopUserPriceLists[0]
                    );
                    if ($tierPrice !== null) {
                        $tierReducedPrice = $tierPrice->getPrice();
                    }
                }
            }

            //next check for custom discounts
            if ($productVariant instanceof CustomDiscountableInterface) {

                $discountPercent = $this->customDiscountFinder->find($productVariant, $context['channel'],
                    $shopUserPriceLists[0]);

                if (null === $discountPercent) {
                    return $tierReducedPrice ?: $unmodifiedPrice;
                } else {
                    //add rabatt
                    $price = $tierReducedPrice ? $tierReducedPrice : $unmodifiedPrice;
                    return (int)($price - ($price * ($discountPercent / 100)));
                }
            }

            return $unmodifiedPrice;
        }

        return $unmodifiedPrice;
    }

    /**
     * return an simple array with price list ids for current user
     * @param ShopUser $shopUser
     * @return array|null
     */
    private function getPriceListsForUser(ShopUser $shopUser): ?array
    {
        $shopUserGroup = $shopUser->getCustomer()->getGroup();
        if (null !== $shopUserGroup) {
            $shopUserPriceLists = [];
            foreach ($shopUserGroup->getPriceLists() as $priceList) {
                $shopUserPriceLists[] = $priceList->getId();
            }
            return $shopUserPriceLists;
        }
        return null;
    }

    /**
     * @param ShopUserInterface|null $shopUser
     */
    public function setShopUser(?ShopUserInterface $shopUser): void
    {
        $this->shopUser = $shopUser;
    }
}
