<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace LifeStyle\Sylius\PricingPlugin\ShopApi\Services\Product\ShowPrice;

use LifeStyle\Sylius\PricingPlugin\Entity\PriceList;
use LifeStyle\Sylius\PricingPlugin\Services\ProductVariantPriceCalculator;
use LifeStyle\Sylius\PricingPlugin\ShopApi\Exception\InvalidRequestParameterException;
use LifeStyle\Sylius\PricingPlugin\ShopApi\Services\HandlerInterface;
use LifeStyle\Sylius\PricingPlugin\ShopApi\Services\Product\ShowPrice\Repository\CommonRepository;
use LifeStyle\Sylius\PricingPlugin\ShopApi\Services\Product\ShowPrice\Request\RequestParserFactory;
use LifeStyle\Sylius\PricingPlugin\ShopApi\Services\Product\ShowPrice\Response\ResponeBuilderFactory;
use LifeStyle\Sylius\PricingPlugin\ShopApi\Services\Product\ShowPrice\ResponseData\ResponseData;
use LifeStyle\Sylius\PricingPlugin\ShopApi\Services\Product\ShowPrice\ResponseData\ResponseDataFactory;
use LifeStyle\Sylius\PricingPlugin\ShopApi\Services\ResponseInterface;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Model\ShopUserInterface;
use Sylius\Component\Product\Model\ProductVariantInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class Handler
 * @package LifeStyle\Sylius\PricingPlugin\ShopApi\Services\Product\ShowPrice
 */
class Handler implements HandlerInterface
{
    /**
     * @var RequestParserFactory
     */
    private $requestParserFactory;

    /**
     * @var ResponseDataFactory
     */
    private $responseDataFactory;

    /**
     * @var ResponeBuilderFactory
     */
    private $responseBuilderFactory;

    /**
     * @var CommonRepository
     */
    private $commonRepository;

    /**
     * @var ProductVariantPriceCalculator
     */
    private $productVariantPriceCalculator;

    /**
     * Handler constructor.
     * @param RequestParserFactory $requestParserFactory
     * @param ResponseDataFactory $responseDataFactory
     * @param ResponeBuilderFactory $responseBuilderFactory
     * @param CommonRepository $commonRepository
     * @param ProductVariantPriceCalculator $productVariantPriceCalculator
     */
    public function __construct(
        RequestParserFactory $requestParserFactory,
        ResponseDataFactory $responseDataFactory,
        ResponeBuilderFactory $responseBuilderFactory,
        CommonRepository $commonRepository,
        ProductVariantPriceCalculator $productVariantPriceCalculator
    ) {
        $this->requestParserFactory = $requestParserFactory;
        $this->responseDataFactory = $responseDataFactory;
        $this->responseBuilderFactory = $responseBuilderFactory;
        $this->commonRepository = $commonRepository;
        $this->productVariantPriceCalculator = $productVariantPriceCalculator;
    }

    /**
     * @param Request $request
     * @return ResponseInterface
     * @throws \Exception
     */
    public function handle(Request $request): ResponseInterface
    {
        $requestData = $this->requestParserFactory->createRequestParser()->parse($request);

        //now check if the channel exists
        $channelModel = $this->commonRepository->getChannelRespository()->findOneByCode($requestData->getChannel());
        if (null === $channelModel) {
            throw new InvalidRequestParameterException('channel not found');
        }

        $productVariantModel = $this->commonRepository->getProductVariantRepository()->findOneBy(['code' => $requestData->getCode()]);
        if (null === $productVariantModel) {
            throw new InvalidRequestParameterException('product variant not found');
        }

        //set responseData
        $responseData = $this->responseDataFactory->responseData();

        //add product data
        $responseData = $this->addProductData($responseData, $productVariantModel);

        //add shopUser
        $shopUser = $this->commonRepository->getUserRepository()->find($requestData->getUserId());
        $responseData = $this->addShopUserData($responseData, $shopUser);

        //add channelData
        $responseData = $this->addChannelData($responseData, $channelModel);

        $this->productVariantPriceCalculator->setShopUser($shopUser);

        //add calculated price
        $responseData = $this->addCalculatedPrice(
            $responseData,
            $productVariantModel,
            $channelModel,
            $requestData->getQuantity()
        );


        //add tierprice data
        if (null !== $responseData->getShopUserData() && !empty($responseData->getShopUserData()->getPriceListId())) {
            $userPriceListId = $responseData->getShopUserData()->getPriceListId();
            $responseData = $this->addTierPriceData(
                $responseData,
                $productVariantModel,
                $channelModel,
                $userPriceListId
            );
        }

        //add custom discounts data
        if (null !== $responseData->getShopUserData() && !empty($responseData->getShopUserData()->getPriceListId())) {
            $userPriceListId = $responseData->getShopUserData()->getPriceListId();
            $responseData = $this->addCustomDiscountsData(
                $responseData,
                $productVariantModel,
                $channelModel,
                $userPriceListId
            );
        }

        return $this->responseBuilderFactory->createResponseBuilder()->buildResponse($responseData);
    }

    /**
     * @param ResponseData $responseData
     * @param ProductVariantInterface $productVariantModel
     * @return ResponseData
     */
    private function addProductData(ResponseData $responseData, ProductVariantInterface $productVariantModel)
    {

        $responseData->setProductVariantCode($productVariantModel->getCode());
        $responseData->setProductVariantId($productVariantModel->getId());
        $responseData->setProductCode($productVariantModel->getProduct()->getCode());
        $responseData->setProductId($productVariantModel->getProduct()->getId());

//        dump($productVariantModel->getTierPrices());
//        dump($productVariantModel->getCustomDiscounts());die();

        return $responseData;
    }

    /**
     * @param ResponseData $responseData
     * @param ShopUserInterface|null $shopUser
     * @return ResponseData
     */
    private function addShopUserData(ResponseData $responseData, ?ShopUserInterface $shopUser)
    {

        if (null !== $shopUser) {
            $userData = $this->responseDataFactory->shopUserData();
            $userData->setUserId($shopUser->getId());
            $userData->setCustomerId($shopUser->getCustomer()->getId());
            $userData->setUserName($shopUser->getUsername());

            $group = $shopUser->getCustomer()->getGroup();

            if (null !== $group) {
                $userData->setGroupId($group->getId());
                $userData->setGroupName($group->getName());
                if (!empty($group->getPriceLists()) && !empty($group->getPriceLists()->first())) {
                    /** @var PriceList $priceListModel */
                    $priceListModel = $group->getPriceLists()->first();
                    $userData->setPriceListId($priceListModel->getId());
                    $userData->setPriceListName($priceListModel->getName());
                    $userData->setPriceListEnabled($priceListModel->getEnabled());
                }
            }
            $responseData->setShopUserData($userData);
        }

        return $responseData;
    }

    /**
     * @param ResponseData $responseData
     * @param ChannelInterface $channel
     * @return ResponseData
     */
    private function addChannelData(ResponseData $responseData, ChannelInterface $channel)
    {
        $channelData = $this->responseDataFactory->channelData();
        $channelData->setChannelId($channel->getId());
        $channelData->setChannelEnabled($channel->isEnabled());
        $channelData->setChannelName($channel->getName());
        $channelData->setChannelCode($channel->getCode());
        $channelData->setBaseCurrencyCode($channel->getBaseCurrency()->getCode());
        $channelData->setBaseCurrencyId($channel->getBaseCurrency()->getId());
        $responseData->setChannelData($channelData);
        //normal prices
        return $responseData;
    }

    /**
     * @param ResponseData $responseData
     * @param ProductVariantInterface $productVariantModel
     * @param ChannelInterface $channel
     * @param int $quanity
     * @return ResponseData
     */
    private function addCalculatedPrice(
        ResponseData $responseData,
        ProductVariantInterface $productVariantModel,
        ChannelInterface $channel,
        int $quanity
    ) {
        $calculatedPriceData = $this->responseDataFactory->calculatedPriceData();

        //tax stuff
        $rate = $this->commonRepository->getTaxRateResolver()->resolve($productVariantModel);
        $calculatedPriceData->setTaxRateAmount($rate->getAmount());
        $calculatedPriceData->setTaxRateAmountAsPercent($rate->getAmountAsPercentage());
        $calculatedPriceData->setTaxIncludedInPrice($rate->isIncludedInPrice());

        //unmodified prices
        $unmodifiedChannelPricing = $productVariantModel->getChannelPricingForChannel($channel);
        $calculatedPriceData->setUnmodifiedUnitPrice($unmodifiedChannelPricing->getPrice());
        $calculatedPriceData->setUnmodifiedOriginalUnitPrice($unmodifiedChannelPricing->getOriginalPrice());

        //now calculate prices
        if ($quanity == 0) {
            $quanity = 1;
        }

        $calculatedPriceData->setQuantity($quanity);

        $context['channel'] = $channel;
        $context['quantity'] = $quanity;
        $calculatedUnitPrice = $this->productVariantPriceCalculator->calculate($productVariantModel, $context);

        $calculatedPriceData->setUnitPrice($calculatedUnitPrice);

        //calculate total price
        $calculatedPriceData->setTotalPrice(($calculatedUnitPrice * $quanity));

        //calculate tax unit amount
        $calculatedPriceData->setTaxRateAmount($this->commonRepository->getTaxCalculator()->calculate($calculatedUnitPrice,
            $rate));

        $responseData->setCalculatedPriceData($calculatedPriceData);

        return $responseData;
    }

    /**
     * @param ResponseData $responseData
     * @param ProductVariantInterface $productVariantModel
     * @param ChannelInterface $channel
     * @param int $userPriceListId
     * @return ResponseData
     */
    private function addTierPriceData(
        ResponseData $responseData,
        ProductVariantInterface $productVariantModel,
        ChannelInterface $channel,
        int $userPriceListId
    ) {
        $tierPrices = $this->commonRepository->getTierPriceFinder()->findAll(
            $productVariantModel,
            $channel,
            $userPriceListId
        );

        foreach ($tierPrices as $tierPrice) {
            $tierPriceData = $this->responseDataFactory->tierPriceData();
            $tierPriceData->setQuantity($tierPrice->getQty());
            $tierPriceData->setPrice($tierPrice->getPrice());
            $tierPriceData->setPriceListId($tierPrice->getPriceList()->getId());

            $responseData->addTierPrice($tierPriceData);
        }

        return $responseData;
    }

    /**
     * @param ResponseData $responseData
     * @param ProductVariantInterface $productVariantModel
     * @param ChannelInterface $channel
     * @param int $userPriceListId
     * @return ResponseData
     */
    private function addCustomDiscountsData(
        ResponseData $responseData,
        ProductVariantInterface $productVariantModel,
        ChannelInterface $channel,
        int $userPriceListId
    ) {
        $customDiscounts = $this->commonRepository->getCustomDiscountFinder()->findAll(
            $productVariantModel,
            $channel,
            $userPriceListId
        );

        foreach ($customDiscounts as $customDiscount) {
            $customDiscountData = $this->responseDataFactory->customDiscountData();
            $customDiscountData->setDiscount($customDiscount->getDiscount());
            $customDiscountData->setPriceListId($userPriceListId);
            $customDiscountData->setType($customDiscount->getType());

            $responseData->addCustomDiscount($customDiscountData);
        }

        return $responseData;
    }
}
