<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace LifeStyle\Sylius\PricingPlugin\Services;

use LifeStyle\Sylius\PricingPlugin\Entity\PriceList;
use LifeStyle\Sylius\PricingPlugin\Entity\ProductVariant;
use LifeStyle\Sylius\PricingPlugin\Entity\TierPrice;
use LifeStyle\Sylius\PricingPlugin\Repository\TierPriceRepositoryInterface;
use LifeStyle\Sylius\PricingPlugin\Services\TierPriceFinder;
use PHPUnit\Framework\TestCase;
use Sylius\Component\Core\Model\ChannelInterface;

class TierPriceFinderTest extends TestCase
{
    /** @var TierPriceFinder */
    private $tierPriceFinder;

    /** @var ChannelInterface */
    private $testChannel;

    /** @var TierPriceRepositoryInterface */
    private $tierPriceRepo;

    public function __construct(
        ?string $name = null,
        array $data = [],
        string $dataName = ''
    ) {
        parent::__construct($name, $data, $dataName);

        $this->tierPriceRepo      = $this->createMock(TierPriceRepositoryInterface::class);
        $this->tierPriceFinder    = new TierPriceFinder($this->tierPriceRepo);

        $this->testChannel = $this->createMock(ChannelInterface::class);
    }

    public function testCalculateWithNotEnoughQuantity()
    {
        //## PREPARE
        $tierPrice = $this->createMock(TierPrice::class);
        $tierPrice->method('getPrice')->willReturn(1);
        $tierPrice->method('getQty')->willReturn(20);

        $productVariant = $this->createMock(ProductVariant::class);
        $this->tierPriceRepo->method('getSortedTierPrices')->willReturn([$tierPrice]);

        //## EXECUTE
        $tierPriceFound = $this->tierPriceFinder->find($productVariant, $this->testChannel, 10, 1);

        //## CHECK
        $this->assertEquals(null, $tierPriceFound);
    }

    public function testCalculateWithOneTierPrice()
    {
        //## PREPARE

        $priceList = $this->createMock(PriceList::class);
        $priceList->method('getId')->willReturn(1);

        $tierPrice = $this->createMock(TierPrice::class);
        $tierPrice->method('getPrice')->willReturn(1);
        $tierPrice->method('getQty')->willReturn(5);
        $tierPrice->method('getPriceList')->willReturn($priceList);

        $productVariant = $this->createMock(ProductVariant::class);
        $this->tierPriceRepo->method('getSortedTierPrices')->willReturn([$tierPrice]);

        //## EXECUTE
        $tierPriceFound = $this->tierPriceFinder->find($productVariant, $this->testChannel, 10, 1);

        //## CHECK
        $this->assertEquals($tierPriceFound, $tierPrice);
    }

    public function testCalculateWithHighestTierPrice()
    {
        //## PREPARE
        $priceList1 = $this->createMock(PriceList::class);
        $priceList1->method('getId')->willReturn(1);

        $priceList2 = $this->createMock(PriceList::class);
        $priceList2->method('getId')->willReturn(2);

        $tierPrice1 = $this->createMock(TierPrice::class);
        $tierPrice1->method('getPrice')->willReturn(500);
        $tierPrice1->method('getQty')->willReturn(10);
        $tierPrice1->method('getPriceList')->willReturn($priceList1);

        $tierPrice2 = $this->createMock(TierPrice::class);
        $tierPrice2->method('getPrice')->willReturn(10);
        $tierPrice2->method('getQty')->willReturn(50);
        $tierPrice1->method('getPriceList')->willReturn($priceList2);

        $productVariant = $this->createMock(ProductVariant::class);
        $this->tierPriceRepo->method('getSortedTierPrices')->willReturn([$tierPrice1, $tierPrice2]);

        //## EXECUTE
        $tierPriceFound = $this->tierPriceFinder->find($productVariant, $this->testChannel, 11, 1);

        //## CHECK
        $this->assertEquals($tierPrice2, $tierPriceFound);
    }
}
