<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace LifeStyle\Sylius\PricingPlugin\Services;

use LifeStyle\Sylius\DataModelPlugin\Entity\TierPriceInterface;
use LifeStyle\Sylius\PricingPlugin\Repository\TierPriceRepositoryInterface;
use LifeStyle\Sylius\PricingPlugin\Traits\TierPriceableInterface;
use Sylius\Component\Core\Model\ChannelInterface;

/**
 * Class TierPriceFinder
 * @package LifeStyle\Sylius\PricingPlugin\Services
 */
class TierPriceFinder implements TierPriceFinderInterface
{
    /** @var TierPriceRepositoryInterface */
    private $tierPriceRepository;

    /**
     * TierPriceFinder constructor.
     * @param TierPriceRepositoryInterface $tierPriceRepository
     */
    public function __construct(TierPriceRepositoryInterface $tierPriceRepository)
    {
        $this->tierPriceRepository = $tierPriceRepository;
    }

    /**
     * @param TierPriceableInterface $tierPriceableEntity
     * @param ChannelInterface $channel
     * @param int $quantity
     * @param int $priceListId
     * @return TierPriceInterface|null
     */
    public function find(
        TierPriceableInterface $tierPriceableEntity,
        ChannelInterface $channel,
        int $quantity,
        int $priceListId
    ): ?TierPriceInterface {
        $possibleTierPrices = $this->tierPriceRepository->getSortedTierPrices($tierPriceableEntity, $channel);

        $cheapestTierPrice = null;
        /** @var TierPriceInterface[] $tierPricesForChannel */
        foreach ($possibleTierPrices as $tierPrice) {
            if ($tierPrice->getQty() > $quantity) {
                break;
            }
            if ($priceListId === $tierPrice->getPriceList()->getId()) {
                $cheapestTierPrice = $tierPrice;
            }
        }

        return $cheapestTierPrice;
    }

    /**
     * @param TierPriceableInterface $tierPriceableEntity
     * @param ChannelInterface $channel
     * @param int $priceListId
     * @return TierPriceInterface[]|null
     */
    public function findAll(
        TierPriceableInterface $tierPriceableEntity,
        ChannelInterface $channel,
        int $priceListId
    ) {
        $possibleTierPrices = $this->tierPriceRepository->getSortedTierPrices($tierPriceableEntity, $channel);
        $tierPriceArray = [];
        /** @var TierPriceInterface[] $tierPricesForChannel */
        foreach ($possibleTierPrices as $tierPrice) {
            if(null !== $tierPrice->getPriceList() && null !== $tierPrice->getPriceList()->getId()){
                if($tierPrice->getPriceList()->getId() === $priceListId){
                    $tierPriceArray[] = $tierPrice;
                }
            }
        }

        return $tierPriceArray;
    }
}
