<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace LifeStyle\Sylius\PricingPlugin\Entity;

use PHPUnit\Framework\TestCase;
use Sylius\Component\Core\Model\ChannelInterface;

/**
 * Class ProductVariantTest
 * @package LifeStyle\Sylius\PricingPlugin\Entity
 */
class ProductVariantTest extends TestCase
{
    /** @var ChannelInterface */
    private $testChannel;

    /** @var ChannelInterface */
    private $otherChannel;

    /**
     * ProductVariantTest constructor.
     * @param string|null $name
     * @param array $data
     * @param string $dataName
     */
    public function __construct(
        ?string $name = null,
        array $data = [],
        string $dataName = ''
    ) {
        parent::__construct($name, $data, $dataName);

        $this->testChannel = $this->createMock(ChannelInterface::class);
        $this->testChannel->method('getId')->willReturn(1);

        $this->otherChannel = $this->createMock(ChannelInterface::class);
        $this->otherChannel->method('getId')->willReturn(2);
    }

    /**
     * @param ChannelInterface $channel
     * @param int $quantity
     * @return TierPriceInterface
     */
    private function createTierPrice(ChannelInterface $channel, int $quantity): TierPriceInterface
    {
        $result = new TierPrice($quantity);
        $result->setChannel($channel);
        $result->setQty($quantity);

        return $result;
    }

    /** @dataProvider data_getTierPricesForChannel
     *
     * @param array $givenTierPrices
     * @param array $expectedTierPrices
     */
    public function test_getTierPricesForChannel(array $givenTierPrices, array $expectedTierPrices): void
    {
        //## PREPARE
        $productVariant = new ProductVariant();
        $productVariant->setTierPrices($givenTierPrices);

        //## EXECUTE
        $resultEntries = $productVariant->getTierPricesForChannel($this->testChannel);

        //## CHECK
        $this->assertEquals(count($resultEntries), count($expectedTierPrices));
        $i = 0;
        foreach ($resultEntries as $entry) {
            /** @var TierPrice $entry */
            $this->assertEquals($expectedTierPrices[$i]->getQty(), $entry->getQty());
            ++$i;
        }
    }

    /**
     * @return array
     */
    public function data_getTierPricesForChannel(): array
    {
        return
            [
                'no tier prices' => [
                    [],
                    [],
                ],
                'one tier price matches' => [
                    // Input
                    [$this->createTierPrice($this->testChannel, 1)],
                    // Expected Output
                    [$this->createTierPrice($this->testChannel, 1)],
                ],
                'one tier price no match' => [
                    // Input
                    [$this->createTierPrice($this->otherChannel, 10)],
                    // Expected Output
                    [],
                ],
                'multiple tier prices' => [
                    // Input
                    [
                        $this->createTierPrice($this->otherChannel, 1),
                        $this->createTierPrice($this->otherChannel, 2),
                        $this->createTierPrice($this->testChannel, 3),
                        $this->createTierPrice($this->otherChannel, 4),
                        $this->createTierPrice($this->testChannel, 5),
                    ],
                    // Expected Output
                    [
                        $this->createTierPrice($this->testChannel, 3),
                        $this->createTierPrice($this->testChannel, 5),
                    ],
                ],
            ];
    }
}
