<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace LifeStyle\Sylius\PricingPlugin\Services;

use Doctrine\Common\Collections\ArrayCollection;
use LifeStyle\Sylius\PricingPlugin\Entity\CustomerGroupInterface;
use LifeStyle\Sylius\PricingPlugin\Entity\PriceList;
use LifeStyle\Sylius\PricingPlugin\Entity\ProductVariant;
use LifeStyle\Sylius\PricingPlugin\Entity\TierPrice;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Sylius\Component\Core\Calculator\ProductVariantPriceCalculatorInterface;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Model\ProductVariant as SyliusProductVariant;
use Sylius\Component\Core\Model\ShopUser;
use Sylius\Component\Customer\Model\CustomerInterface;

class ProductVariantPriceCalculatorTest extends TestCase
{
    /** @var ProductVariantPriceCalculatorInterface|MockObject */
    private $basePriceCalculator;

    /** @var ProductVariantPriceCalculatorInterface|MockObject */
    private $priceCalculator;

    /** @var TierPriceFinderInterface|MockObject */
    private $tierPriceFinder;

    /** @var CustomDiscountFinderInterface|MockObject */
    private $customDiscountFinder;

    /** @var ShopUserManagerInterface|MockObject */
    private $shopUserManager;

    public function testCalculateWithNonTierpriceable()
    {
        //## PREPARE
        $productVariant = $this->createMock(SyliusProductVariant::class);
        $testChannel = $this->createMock(ChannelInterface::class);

        //## EXECUTE
        $price = $this->priceCalculator->calculate($productVariant, ['channel' => $testChannel, 'quantity' => 10]);

        //## CHECK
        $this->assertEquals(-1, $price);
    }

    public function testCalculatePriceWithEmptyTierPrices()
    {
        //## PREPARE
        $productVariant = $this->createMock(ProductVariant::class);
        $testChannel = $this->createMock(ChannelInterface::class);

        $this->tierPriceFinder->method('find')->willReturn(null);

        //## EXECUTE
        $result = $this->priceCalculator->calculate($productVariant, ['channel' => $testChannel, 'quantity' => 10]);

        //## CHECK
        $this->assertEquals(-1, $result);
    }

    /**
     * @param int $discountPercentage
     * @param int $tierPriceQty
     * @param int $tierPrice
     * @param int $calcQty
     * @param int $calcResult
     * @dataProvider getCalculateData
     */
    public function testCalculatePriceWithTierPrices(
        int $discountPercentage,
        int $tierPriceQty,
        int $tierPrice,
        int $calcQty,
        int $calcResult
    ) {
        //## PREPARE
        $productVariant = $this->createMock(ProductVariant::class);
        $testChannel = $this->createMock(ChannelInterface::class);

        // We would like to test discount calculation
        if ($discountPercentage >= 0) {
            $this->customDiscountFinder->method('find')->willReturn($discountPercentage);
        }

        // Test calculator with tier price
        if ($tierPriceQty >= 0) {
            $this->tierPriceFinder->method('find')->willReturn(new TierPrice($tierPriceQty, $tierPrice));
        }

        //## EXECUTE
        $result = $this->priceCalculator->calculate($productVariant, ['channel' => $testChannel, 'quantity' => $calcQty]);

        //## CHECK
        $this->assertEquals($calcResult, $result);
    }

    protected function setUp(): void
    {
        $this->basePriceCalculator = $this->createMock(ProductVariantPriceCalculatorInterface::class);
        $this->basePriceCalculator->method('calculate')->willReturn(-1); // To indicate no tier prices

        $this->tierPriceFinder = $this->createMock(TierPriceFinderInterface::class);

        $this->customDiscountFinder = $this->createMock(CustomDiscountFinderInterface::class);

        $priceList = $this->createMock(PriceList::class);
        $priceList->method('getId')->willReturn(1);

        $group = $this->createMock(CustomerGroupInterface::class);
        $group->method('getPriceLists')->willReturn(new ArrayCollection([$priceList]));

        $customer = $this->createMock(CustomerInterface::class);
        $customer->method('getGroup')->willReturn($group);

        $shopUser = $this->createMock(ShopUser::class);
        $shopUser->method('getCustomer')->willReturn($customer);

        $this->shopUserManager = $this->createMock(ShopUserManagerInterface::class);
        $this->shopUserManager->method('getShopUser')->willReturn($shopUser);

        $this->priceCalculator = new ProductVariantPriceCalculator(
            $this->basePriceCalculator,
            $this->tierPriceFinder,
            $this->customDiscountFinder,
            $this->shopUserManager
        );
    }

    /**
     * Calculate several scenarios
     * @return array
     */
    public function getCalculateData(): array
    {
        return [
            // Round half up
            [
                25,    // Get 25% off
                2,     // Tier price min-quantity: 2 - not checked in this test
                22050, // Tier price: 220,50
                10,    // Buy 10 pieces
                16538, // 220,50 - 25% = 165,375 => 165,38
            ],
            // Round down
            [
                24,    // Get 24% off
                2,     // Tier price min-quantity: 2 - not checked in this test
                22048, // Tier price: 220,48
                10,    // Buy 10 pieces
                16756, // 220,48 - 24% = 167,5648 => 167,56
            ],
            // No need to round
            [
                20,    // Get 20% off
                2,     // Tier price min-quantity: 2 - not checked in this test
                59900, // Tier price: 599,00
                10,    // Buy 10 pieces
                47920, // 599,00 - 20% = 479,20
            ],
        ];
    }
}
