<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\DependencyInjection;

use Lifestyle\Sylius\Product\Configuration\AssetConfiguration;
use Lifestyle\Sylius\Product\Entity\ProductImage;
use Lifestyle\Sylius\Product\Entity\ProductImageInterface;
use Lifestyle\Sylius\Product\MessageHandler\AssetPublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\AssetUnpublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\DefaultUnpublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\ProductPublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\TaxonPublishHandler;
use Symfony\Component\DependencyInjection\Extension\PrependExtensionInterface;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;
use Symfony\Component\DependencyInjection\Loader\XmlFileLoader;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\Config\FileLocator;

/**
 * Class LifestyleSyliusProductExtension
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\DependencyInjection
 */
class LifestyleSyliusProductExtension extends Extension implements PrependExtensionInterface
{
    /**
     * Loads a specific configuration.
     *
     * @param array $configs
     * @param ContainerBuilder $container
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $configuration = $this->getConfiguration($configs, $container);
        $config = $this->processConfiguration($configuration, $configs);

        (new XmlFileLoader($container, new FileLocator(dirname(__DIR__) . '/Resources/config')))->load('services.xml');

        $this->configureAssetConfiguration($config, $container);
        $this->configureAssetPublishHandler($config, $container);
        $this->configureAssetUnpublishHandler($config, $container);
        $this->configureDefaultUnpublishHandler($config, $container);
        $this->configureProductPublishHandler($config, $container);
        $this->configureTaxonPublishHandler($config, $container);
    }

    /**
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureAssetConfiguration(array $config, ContainerBuilder $container)
    {
        if (!$container->has(AssetConfiguration::class)) {
            return;
        }

        $definition = $container->findDefinition(AssetConfiguration::class);
        $definition->setArgument(0, $config['assets']['package_name']);
    }

    /**
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureAssetPublishHandler(array $config, ContainerBuilder $container)
    {
        if (!$container->has(AssetPublishHandler::class)) {
            return;
        }

        $definition = $container->findDefinition(AssetPublishHandler::class);

        $responsibleEvents = isset($config['assets']['responsible_events']['publish']) ? $config['assets']['responsible_events']['publish'] : [];
        foreach ($responsibleEvents as $responsibleEvent) {
            $definition->addMethodCall('addResponsibleEvent', [$responsibleEvent]);
        }
    }

    /**
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureAssetUnpublishHandler(array $config, ContainerBuilder $container)
    {
        if (!$container->has(AssetUnpublishHandler::class)) {
            return;
        }

        $definition = $container->findDefinition(AssetUnpublishHandler::class);

        $responsibleEvents = isset($config['assets']['responsible_events']['unpublish']) ? $config['assets']['responsible_events']['unpublish'] : [];
        foreach ($responsibleEvents as $responsibleEvent) {
            $definition->addMethodCall('addResponsibleEvent', [$responsibleEvent]);
        }
    }

    /**
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureDefaultUnpublishHandler(array $config, ContainerBuilder $container)
    {
        if (!$container->has(DefaultUnpublishHandler::class)) {
            return;
        }

        $definition = $container->findDefinition(DefaultUnpublishHandler::class);

        $responsibleEvents = array_merge(
            isset($config['products']['responsible_events']['unpublish']) ? $config['products']['responsible_events']['unpublish'] : [],
            isset($config['taxon']['responsible_events']['unpublish']) ? $config['taxon']['responsible_events']['unpublish'] : []
        );
        foreach ($responsibleEvents as $responsibleEvent) {
            $definition->addMethodCall('addResponsibleEvent', [$responsibleEvent]);
        }
    }

    /**
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureProductPublishHandler(array $config, ContainerBuilder $container)
    {
        if (!$container->has(ProductPublishHandler::class)) {
            return;
        }

        $definition = $container->findDefinition(ProductPublishHandler::class);

        $responsibleEvents = isset($config['products']['responsible_events']['publish']) ? $config['products']['responsible_events']['publish'] : [];
        foreach ($responsibleEvents as $responsibleEvent) {
            $definition->addMethodCall('addResponsibleEvent', [$responsibleEvent]);
        }
    }

    /**
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureTaxonPublishHandler(array $config, ContainerBuilder $container)
    {
        if (!$container->has(TaxonPublishHandler::class)) {
            return;
        }

        $definition = $container->findDefinition(TaxonPublishHandler::class);

        $responsibleEvents = isset($config['taxon']['responsible_events']['publish']) ? $config['taxon']['responsible_events']['publish'] : [];
        foreach ($responsibleEvents as $responsibleEvent) {
            $definition->addMethodCall('addResponsibleEvent', [$responsibleEvent]);
        }
    }

    /**
     * {@inheritDoc}
     */
    public function prepend(ContainerBuilder $container)
    {
        $config = array_merge(...$container->getExtensionConfig('sylius_core'));

        // do not register if resources not configured.
        if (!empty($config['resources'])) {
            $container->prependExtensionConfig('sylius_core', [
                'resources' => [
                    'product_image' => [
                        'classes' => [
                            'model' => ProductImage::class,
                        ],
                    ],
                ],
            ]);
        }
    }
}
