<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\ProductHelper;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityRepository;
use Lifestyle\Sylius\Product\Configuration\AssetConfiguration;
use Lifestyle\Sylius\Product\Entity\ProductImage;
use Lifestyle\Sylius\Product\Exception\InvalidArgumentException;
use Lifestyle\Sylius\Product\Exception\ObjectNotFoundException;
use Sylius\Component\Core\Model\Product;
use Sylius\Component\Core\Model\ProductVariant;
use Sylius\Component\Resource\Factory\Factory as ImageFactory;
use Symfony\Component\Asset\Packages;

/**
 * Class ProductImageHelper
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\ProductHelper
 */
class ProductImageHelper
{
    /**
     * @var AssetConfiguration
     */
    private $assetConfiguration;

    /**
     * @var ImageFactory
     */
    private $imageFactory;

    /**
     * @var EntityRepository
     */
    private $imageRepository;

    /**
     * @var EntityManager
     */
    private $imageManager;

    /**
     * @var EntityRepository
     */
    private $productRepository;

    /**
     * @var EntityManager
     */
    private $productManager;

    /**
     * @var EntityRepository
     */
    private $productVariantRepository;

    /**
     * @var Packages
     */
    private $packages;

    /**
     * ProductImageHelper constructor.
     * @param AssetConfiguration $assetConfiguration
     * @param ImageFactory $imageFactory
     * @param EntityRepository $imageRepository
     * @param EntityManager $imageManager
     * @param EntityRepository $productRepository
     * @param EntityManager $productManager
     * @param EntityRepository $productVariantRepository
     * @param Packages $packages
     */
    public function __construct(
        AssetConfiguration $assetConfiguration,
        ImageFactory $imageFactory,
        EntityRepository $imageRepository,
        EntityManager $imageManager,
        EntityRepository $productRepository,
        EntityManager $productManager,
        EntityRepository $productVariantRepository,
        Packages $packages
    ) {
        $this->assetConfiguration = $assetConfiguration;
        $this->imageFactory = $imageFactory;
        $this->imageRepository = $imageRepository;
        $this->imageManager = $imageManager;
        $this->productRepository = $productRepository;
        $this->productManager = $productManager;
        $this->productVariantRepository = $productVariantRepository;
        $this->packages = $packages;
    }

    /**
     * @param string $resourceId
     * @throws \Doctrine\ORM\ORMException
     */
    public function updateImage(string $resourceId)
    {
        $resourceId = (string)$resourceId;
        if (0 === strlen($resourceId)) {
            throw new InvalidArgumentException(
                'Cannot proceed image-update! Image-Id not set in message.'
            );
        }

        $path = $this->getImagePath($resourceId);

        /** @var ProductImage[] $images */
        $images = $this->imageRepository->findBy(['resourceId' => $resourceId]);
        foreach ($images as $image) {
            $image->setPath($path);
            $this->imageManager->persist($image);
        }
        if (0 < count($images)) {
            $this->imageManager->flush();
        }
    }

    /**
     * @param string $resourceId
     * @throws \Doctrine\ORM\ORMException
     */
    public function deleteImage(string $resourceId)
    {
        $resourceId = (string)$resourceId;
        if (0 === strlen($resourceId)) {
            throw new InvalidArgumentException(
                'Cannot proceed image-update! Image-Id not set in message.'
            );
        }
        /** @var ProductImage[] $images */
        $images = $this->imageRepository->findBy(['resourceId' => $resourceId]);
        foreach ($images as $image) {
            $this->imageManager->remove($image);
        }
        if (0 < count($images)) {
            $this->imageManager->flush();
        }
    }

    /**
     * @param string $productCode
     * @param string $imageResourceId
     * @throws \Doctrine\ORM\ORMException
     */
    public function createOrUpdateProductImage(string $productCode, string $imageResourceId)
    {
        $this->createOrUpdateProductImageByProduct($this->getProduct($productCode), $imageResourceId);
    }

    /**
     * @param string $productCode
     * @param string $productVariantCode
     * @param string $imageResourceId
     * @throws \Doctrine\ORM\ORMException
     */
    public function createOrUpdateProductVariantImage(string $productCode, string $productVariantCode, string $imageResourceId)
    {
        $product = $this->getProduct($productCode);
        $image = $this->createOrUpdateProductImageByProduct($product, $imageResourceId);

        $productVariant = $this->productVariantRepository->findOneBy(['product' => $product, 'code' => $productVariantCode]);
        if (!$productVariant instanceof ProductVariant) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed image-update! ProductVariant (code: %s) not found.',
                $productVariantCode
            ));
        }

        if (!$image->hasProductVariant($productVariant)) {
            $image->addProductVariant($productVariant);
            $this->imageManager->persist($image);
            $this->imageManager->flush($image);
        }
    }

    /**
     * @param string $resourceId
     * @return string
     */
    private function getImagePath(string $resourceId): string
    {
        $path = $this->packages->getUrl($resourceId, $this->assetConfiguration->getPackageName());
        if (0 === strlen($path)) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed image-update! Image-path (id: %s) not found.',
                $resourceId
            ));
        }

        return $path;
    }

    /**
     * @param Product $product
     * @param string $imageResourceId
     * @param string $imagePath
     * @return ProductImage
     * @throws \Doctrine\ORM\ORMException
     */
    private function createImage(Product $product, string $imageResourceId, string $imagePath): ProductImage
    {
        /** @var ProductImage $image */
        $image = $this->imageFactory->createNew();
        $image->setResourceId($imageResourceId);
        $image->setPath($imagePath);
        $image->setOwner($product);
        $this->imageManager->persist($image);
        $this->imageManager->flush($image);

        return $image;
    }

    /**
     * @param string $productCode
     * @return Product
     */
    private function getProduct(string $productCode): Product
    {
        $product = $this->productRepository->findOneBy(['code' => $productCode]);
        if (!$product instanceof Product) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed image-update! Product (code: %s) not found.',
                $productCode
            ));
        }

        return $product;
    }

    /**
     * @param Product $product
     * @param string $imageResourceId
     * @return ProductImage
     * @throws \Doctrine\ORM\ORMException
     */
    private function createOrUpdateProductImageByProduct(Product $product, string $imageResourceId): ProductImage
    {
        $path = $this->getImagePath($imageResourceId);
        $image = $this->imageRepository->findOneBy(['owner' => $product->getId(), 'resourceId' => $imageResourceId]);
        if (null === $image) {
            $image = $this->createImage($product, $imageResourceId, $path);
            $product->addImage($image);
            $this->productManager->persist($product);
            $this->productManager->flush($product);
        } else {
            $image->setPath($path);
            $this->imageManager->persist($image);
            $this->imageManager->flush($image);
        }

        return $image;
    }
}
