<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\Request\Validator;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\Product\Request\Model\ProductPublish\Factory;
use Symfony\Component\Validator\Exception\ValidatorException;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Class ProductPublishValidator
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\Request\Validator
 */
class ProductPublishValidator
{
    /**
     * @var Factory
     */
    private $modelFactory;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * ProductPublishValidator constructor.
     * @param Factory $modelFactory
     * @param ValidatorInterface $validator
     */
    public function __construct(Factory $modelFactory, ValidatorInterface $validator)
    {
        $this->modelFactory = $modelFactory;
        $this->validator = $validator;
    }

    /**
     * Do a minimal input validation
     *
     * @param DataCollectorInterface $dataCollector
     * @throws ValidatorException
     */
    public function validate(DataCollectorInterface $dataCollector): void
    {
        $product = $this->modelFactory->product();
        $product->setId($dataCollector->getItemValue('id'));
        $product->setParentId($dataCollector->getItemValue('parentId'));
        $product->setOnHand($dataCollector->getItemValue('onHand'));
        $product->setChannels($dataCollector->getItemValue('channels'));
        $product->setPrices($dataCollector->getItemValue('channelPrices'));

        foreach ($dataCollector->getCollectedLocales() as $locale) {
            $localizedFields = $this->modelFactory->localizedFields();
            $localizedFields->setLocale($locale);
            $localizedFields->setName($dataCollector->getItemValue('name', $locale));
            $localizedFields->setSlug($dataCollector->getItemValue('slug', $locale));
            $product->addLocalizedFields($localizedFields);
        }

        if ($dataCollector->hasItem('attributes')) {
            $attributes = $dataCollector->getItemValue('attributes');
            foreach ((is_array($attributes) ? $attributes : [$attributes]) as $attributeCode) {
                $attribute = $this->modelFactory->attribute();
                $attribute->setName($attributeCode);
                $product->addAttribute($attribute);
            }
        }

        if ($dataCollector->hasItem('options')) {
            $options = $dataCollector->getItemValue('options');
            foreach ((is_array($options) ? $options : [$options]) as $optionCode) {
                $option = $this->modelFactory->option();
                $option->setName($optionCode);
                $product->addOption($option);
            }
        }

        $validationGroup = $dataCollector->getItemValue('type') === 'variant' ? 'variant' : 'product';
        $violations = $this->validator->validate($product, null, $validationGroup);
        if (0 < count($violations)) {
            $errorMessages = [];
            foreach ($violations as $violation) {
                $errorMessages[] = (string)$violation;
            }
            throw new ValidatorException(sprintf(
                'Bad message format: %s',
                join(', ', $errorMessages)
            ));
        }
    }
}
