<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\MessageHandler;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use Lifestyle\Sylius\Product\Request\Validator\ProductUnpublishValidator;

/**
 * Class ProductUnpublishHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\MessageHandler
 */
class ProductUnpublishHandler implements HandlerInterface
{
    use MessageResponsibilityTrait;
    use MessageHandlerTrait;

    /**
     * @var ProductUnpublishValidator
     */
    private $productValidator;

    /**
     * ProductUnpublishHandler constructor.
     * @param ProductUnpublishValidator $productValidator
     */
    public function __construct(ProductUnpublishValidator $productValidator)
    {
        $this->productValidator = $productValidator;
    }

    /**
     * Products will not be deleted but disabled
     *
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        // Do not really delete products but disable them - hopefully this does not cause any problems
        return $this->disableProduct($dataCollector);
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return
            in_array($dataCollector->getItemValue('event'), $this->responsibleEvents) &&
            'variant' !== $dataCollector->getItemValue('type');
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function disableProduct(DataCollectorInterface $dataCollector): bool
    {
        if (!$dataCollector->hasItem('enabled') || $dataCollector->getItemValue('enabled')) {
            $dataCollector = clone $dataCollector;
            $dataCollector->addItem('enabled', false);
        }

        $this->productValidator->validate($dataCollector);

        $processed = false;
        foreach ($this->messageHandler as $handler) {
            if ($handler->isResponsible($dataCollector)) {
                $handler->createOrUpdate($dataCollector);
                $processed = true;
            }
        }

        return $processed;
    }
}
