<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\ProductHelper;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\ORMException;
use Lifestyle\Sylius\Product\Asset\VersionStrategy\JsonManifestVersionStrategy;
use Lifestyle\Sylius\Product\Configuration\AssetConfiguration;
use Lifestyle\Sylius\Product\Entity\ProductImage;
use Lifestyle\Sylius\Product\Exception\ObjectNotFoundException;
use Lifestyle\Sylius\Product\Exception\DatabaseException;
use Sylius\Component\Core\Model\Product;
use Sylius\Component\Core\Model\ProductVariant;
use Sylius\Component\Resource\Factory\Factory as ImageFactory;
use Symfony\Component\Asset\Packages;

/**
 * Class ProductImageHelper
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\ProductHelper
 */
class ProductImageHelper
{
    use DatabaseTrait;

    /**
     * @var AssetConfiguration
     */
    private $assetConfiguration;

    /**
     * @var JsonManifestVersionStrategy
     */
    private $assetVersionStrategy;

    /**
     * @var ImageFactory
     */
    private $imageFactory;

    /**
     * @var EntityManager
     */
    private $imageManager;

    /**
     * @var EntityManager
     */
    private $productManager;

    /**
     * @var EntityManager
     */
    private $productVariantManager;

    /**
     * @var EntityRepository
     */
    private $productRepository;

    /**
     * @var Packages
     */
    private $packages;

    /**
     * ProductImageHelper constructor.
     * @param AssetConfiguration $assetConfiguration
     * @param JsonManifestVersionStrategy $assetVersionStrategy
     * @param ImageFactory $imageFactory
     * @param EntityManager $imageManager
     * @param EntityManager $productManager
     * @param EntityManager $productVariantManager
     * @param EntityRepository $productRepository
     * @param Packages $packages
     */
    public function __construct(
        AssetConfiguration $assetConfiguration,
        JsonManifestVersionStrategy $assetVersionStrategy,
        ImageFactory $imageFactory,
        EntityManager $imageManager,
        EntityManager $productManager,
        EntityManager $productVariantManager,
        EntityRepository $productRepository,
        Packages $packages
    ) {
        $this->assetConfiguration = $assetConfiguration;
        $this->assetVersionStrategy = $assetVersionStrategy;
        $this->imageFactory = $imageFactory;
        $this->imageManager = $imageManager;
        $this->productManager = $productManager;
        $this->productVariantManager = $productVariantManager;
        $this->productRepository = $productRepository;
        $this->packages = $packages;
    }

    /**
     * @param string $productCode
     * @throws ORMException
     * @throws DatabaseException
     */
    public function deleteProductImages(string $productCode)
    {
        $this->updateProductImages($productCode, []);
    }

    /**
     * @param string $productCode
     * @param array $imageResourceIds
     * @throws ORMException
     * @throws DatabaseException
     */
    public function updateProductImages(string $productCode, array $imageResourceIds)
    {
        $this->assetVersionStrategy->clearCache();
        $this->imageManager = $this->getClearedEntityManager($this->imageManager);

        $product = $this->getProduct($productCode);
        $productImages = $this->getProductImages($product);

        foreach (array_diff($imageResourceIds, array_keys($productImages)) as $imageResourceId) {
            $path = $this->getImagePath($imageResourceId);
            $image = $this->createImage($product, $imageResourceId, $path, true);
            $this->imageManager->persist($image);
            $product->addImage($image);
            $productImages[$image->getResourceId()] = $image;
        }

        foreach (array_diff_key($productImages, array_flip($imageResourceIds)) as $image) {
            /** @var ProductImage $image */
            if (!$image->hasProductVariants()) {
                $product->removeImage($image);
                $this->imageManager->remove($image);
            }
        }

        foreach ($productImages as $imageResourceId => $image) {
            $image->setRemovable(!in_array($imageResourceId, $imageResourceIds));
        }

        $this->flush($this->imageManager);
        $this->flush($this->productManager);
    }

    /**
     * @param string $productCode
     * @param string $productVariantCode
     * @throws ORMException
     * @throws DatabaseException
     */
    public function deleteProductVariantImages(string $productCode, string $productVariantCode)
    {
        $this->updateProductVariantImages($productCode, $productVariantCode, []);
    }

    /**
     * @param string $productCode
     * @param string $productVariantCode
     * @param array $imageResourceIds
     * @throws ORMException
     * @throws DatabaseException
     */
    public function updateProductVariantImages(string $productCode, string $productVariantCode, array $imageResourceIds)
    {
        $this->assetVersionStrategy->clearCache();
        $this->imageManager = $this->getClearedEntityManager($this->imageManager);

        $product = $this->getProduct($productCode);
        $productVariant = $this->getProductVariant($productVariantCode, $product);
        $productImages = $this->getProductImages($product);

        foreach (array_diff($imageResourceIds, array_keys($productImages)) as $imageResourceId) {
            $path = $this->getImagePath($imageResourceId);
            $image = $this->createImage($product, $imageResourceId, $path, true);
            $this->imageManager->persist($image);
            $product->addImage($image);
            $productImages[$image->getResourceId()] = $image;
        }

        foreach (array_diff_key($productImages, array_flip($imageResourceIds)) as $image) {
            /** @var ProductImage $image */
            $image->removeProductVariant($productVariant);
            if (!$image->hasProductVariants() && $image->isRemovable()) {
                $product->removeImage($image);
                $this->imageManager->remove($image);
            }
        }

        foreach (array_intersect_key($productImages, array_flip($imageResourceIds)) as $image) {
            if (!$image->hasProductVariant($productVariant)) {
                $image->addProductVariant($productVariant);
            }
        }

        $this->flush($this->imageManager);
        $this->flush($this->productManager);
        $this->flush($this->productVariantManager);
    }

    /**
     * @param string $productCode
     * @return Product
     */
    private function getProduct(string $productCode): Product
    {
        $product = $this->productRepository->findOneBy(['code' => $productCode]);
        if (!$product instanceof Product) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed image-update! Product (code: %s) not found.',
                $productCode
            ));
        }

        return $product;
    }

    /**
     * @param string $productVariantCode
     * @param Product $product
     * @return ProductVariant
     */
    private function getProductVariant(string $productVariantCode, Product $product): ProductVariant
    {
        $productVariant = $product->getVariants()
            ->filter(function (ProductVariant $productVariant) use ($productVariantCode) {
                return $productVariant->getCode() === $productVariantCode;
            })
            ->first();
        if (!$productVariant instanceof ProductVariant) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed image-update! ProductVariant (code: %s) not found for product (code: %s).',
                $productVariantCode,
                $product->getCode()
            ));
        }
        return $productVariant;
    }

    /**
     * @param Product $product
     * @return ProductImage[]|false
     */
    private function getProductImages(Product $product)
    {
        $productImages = $product->getImages()->toArray();
        $productImageResourceIds = array_map(
            function (ProductImage $image) {
                return $image->getResourceId();
            },
            $productImages
        );
        return array_combine($productImageResourceIds, $productImages);
    }

    /**
     * @param Product $product
     * @param string $imageResourceId
     * @param string $imagePath
     * @param bool $isRemovable
     * @return ProductImage
     * @throws ORMException
     */
    private function createImage(
        Product $product,
        string $imageResourceId,
        string $imagePath,
        bool $isRemovable
    ): ProductImage {
        /** @var ProductImage $image */
        $image = $this->imageFactory->createNew();
        $image->setResourceId($imageResourceId);
        $image->setRemovable($isRemovable);
        $image->setPath($imagePath);
        $image->setOwner($product);
        $this->imageManager->persist($image);

        return $image;
    }

    /**
     * @param string $resourceId
     * @return string
     */
    private function getImagePath(string $resourceId): string
    {
        $path = $this->packages->getUrl($resourceId, $this->assetConfiguration->getPackageName());
        if (0 === strlen($path)) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed image-update! Image-path (id: %s) not found.',
                $resourceId
            ));
        }

        return $path;
    }
}
