<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\EventListener;

use Doctrine\ORM\Event\OnFlushEventArgs;
use Doctrine\ORM\Event\PostFlushEventArgs;
use Lifestyle\Sylius\Product\Entity\ProductImage;
use Lifestyle\Sylius\Product\ProductHelper\ProductVariantExtensionDetector;
use Liip\ImagineBundle\Imagine\Cache\CacheManager;
use Liip\ImagineBundle\Imagine\Filter\FilterManager;
use Sylius\Component\Core\Model\ImageInterface;
use Sylius\Component\Core\Uploader\ImageUploaderInterface;

/**
 * Class ImagesRemoveListener
 *
 * This class has been build as a copy of Sylius\Bundle\CoreBundle\EventListener\ImagesRemoveListener
 * As this class is marked as 'final', it was not possible to extend it.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\EventListener
 */
class ImagesRemoveListener
{
    /** @var ImageUploaderInterface */
    private $imageUploader;

    /** @var CacheManager */
    private $cacheManager;

    /** @var FilterManager */
    private $filterManager;

    /** @var string[] */
    private $imagesToDelete = [];

    /**
     * ImagesRemoveListener constructor.
     * @param ImageUploaderInterface $imageUploader
     * @param CacheManager $cacheManager
     * @param FilterManager $filterManager
     */
    public function __construct(
        ImageUploaderInterface $imageUploader,
        CacheManager $cacheManager,
        FilterManager $filterManager
    ) {
        $this->imageUploader = $imageUploader;
        $this->cacheManager = $cacheManager;
        $this->filterManager = $filterManager;
    }

    /**
     * @param OnFlushEventArgs $event
     */
    public function onFlush(OnFlushEventArgs $event): void
    {
        foreach ($event->getEntityManager()->getUnitOfWork()->getScheduledEntityDeletions() as $entityDeletion) {
            if (!$entityDeletion instanceof ImageInterface) {
                continue;
            }

            // Do not delete external files
            if (
                $entityDeletion instanceof ProductImage &&
                null !== $entityDeletion->getResourceId()
            ) {
                continue;
            }

            if (!in_array($entityDeletion->getPath(), $this->imagesToDelete)) {
                $this->imagesToDelete[] = $entityDeletion->getPath();
            }
        }
    }

    /**
     * @param PostFlushEventArgs $event
     */
    public function postFlush(PostFlushEventArgs $event): void
    {
        foreach ($this->imagesToDelete as $key => $imagePath) {
            $this->imageUploader->remove($imagePath);
            $this->cacheManager->remove($imagePath, array_keys($this->filterManager->getFilterConfiguration()->all()));
            unset($this->imagesToDelete[$key]);
        }
    }
}
