<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\ProductHelper;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\ORMException;
use Lifestyle\Sylius\Product\Configuration\AssetConfiguration;
use Lifestyle\Sylius\Product\Exception\ObjectNotFoundException;
use Lifestyle\Sylius\Product\Exception\DatabaseException;
use Lifestyle\Sylius\Product\Entity\Product;
use Sylius\Component\Core\Model\ProductVariant;
use Sylius\Component\Resource\Factory\Factory as ImageFactory;
use Lifestyle\DataCollector\DataCollectorInterface;

/**
 * Class ProductAssetHelper
 * @package Lifestyle\Sylius\Product\ProductHelper
 */
class ProductAssetHelper
{
    use DatabaseTrait;

    /**
     * @var AssetConfiguration
     */
    private $assetConfiguration;

    /**
     * @var ImageFactory
     */
    private $imageFactory;

    /**
     * @var EntityManager
     */
    private $imageManager;

    /**
     * @var EntityManager
     */
    private $productManager;

    /**
     * @var EntityManager
     */
    private $productVariantManager;

    /**
     * @var EntityRepository
     */
    private $productRepository;

    /**
     * @var EntityRepository
     */
    private $assetStorageRepository;

    /**
     * ProductAssetHelper constructor.
     * @param AssetConfiguration $assetConfiguration
     * @param ImageFactory $imageFactory
     * @param EntityManager $imageManager
     * @param EntityManager $productManager
     * @param EntityManager $productVariantManager
     * @param EntityRepository $productRepository
     * @param EntityRepository $assetStorageRepository
     */
    public function __construct(
        AssetConfiguration $assetConfiguration,
        ImageFactory $imageFactory,
        EntityManager $imageManager,
        EntityManager $productManager,
        EntityManager $productVariantManager,
        EntityRepository $productRepository,
        EntityRepository $assetStorageRepository
    ) {
        $this->assetConfiguration = $assetConfiguration;
        $this->imageFactory = $imageFactory;
        $this->imageManager = $imageManager;
        $this->productManager = $productManager;
        $this->productVariantManager = $productVariantManager;
        $this->productRepository = $productRepository;
        $this->assetStorageRepository = $assetStorageRepository;
    }

    /**
     * @param string $productCode
     * @throws ORMException
     * @throws DatabaseException
     */
    public function deleteProductImages(string $productCode)
    {
        $this->updateProductImages($productCode, []);
    }

    /**
     * @param string $productCode
     * @param array $imageResourceIds
     * @throws ORMException
     * @throws DatabaseException
     */
    public function updateProductAssets(string $productCode, DataCollectorInterface $dataCollector)
    {
        $assetIdentifier = $this->assetConfiguration->getProductToAssetIdentifier();
        $assetResourceIds = $dataCollector->hasItem($assetIdentifier) ? $dataCollector->getItemValue($assetIdentifier) : [];

        $product = $this->getProduct($productCode);

        //first remove all assets from product
        if($product->hasAssets()){
            foreach ($product->getAssets() as $asset){
                $product->removeAsset($asset);
            }
        }

        //search asset
        if(!empty($assetResourceIds)){
            foreach($assetResourceIds as $assetResourceId){
                $assetStorageEntity = $this->assetStorageRepository->findOneBy(['resourceId' => $assetResourceId]);
                if(null !== $assetStorageEntity){
                    $product->addAsset($assetStorageEntity);
                }
            }
        }

        $this->productManager->persist($product);
        $this->flush($this->productManager);
    }

    /**
     * @param string $productCode
     * @return Product
     */
    private function getProduct(string $productCode): Product
    {
        $product = $this->productRepository->findOneBy(['code' => $productCode]);
        if (!$product instanceof Product) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed image-update! Product (code: %s) not found.',
                $productCode
            ));
        }

        return $product;
    }

    /**
     * @param string $productVariantCode
     * @param Product $product
     * @return ProductVariant
     */
    private function getProductVariant(string $productVariantCode, Product $product): ProductVariant
    {
        $productVariant = $product->getVariants()
            ->filter(function (ProductVariant $productVariant) use ($productVariantCode) {
                return $productVariant->getCode() === $productVariantCode;
            })
            ->first();
        if (!$productVariant instanceof ProductVariant) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed image-update! ProductVariant (code: %s) not found for product (code: %s).',
                $productVariantCode,
                $product->getCode()
            ));
        }
        return $productVariant;
    }

}
