<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\ProductHelper;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityRepository;
use Lifestyle\Sylius\Product\Exception\ObjectNotFoundException;
use Sylius\Component\Core\Model\Product;
use Sylius\Component\Core\Model\ProductVariant;

/**
 * Class ProductVariantHelper
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\ProductHelper
 */
class ProductVariantHelper
{
    use DatabaseTrait;

    /**
     * @var EntityRepository
     */
    private $productRepository;

    /**
     * @var EntityManager
     */
    private $productVariantManager;

    /**
     * @var ProductVariantExtensionDetector
     */
    private $extensionDetector;

    /**
     * ProductVariantHelper constructor.
     * @param EntityRepository $productRepository
     * @param EntityManager $productVariantManager
     * @param ProductVariantExtensionDetector $extensionDetector
     */
    public function __construct(
        EntityRepository $productRepository,
        EntityManager $productVariantManager,
        ProductVariantExtensionDetector $extensionDetector
    ) {
        $this->productRepository = $productRepository;
        $this->productVariantManager = $productVariantManager;
        $this->extensionDetector = $extensionDetector;
    }

    /**
     * @param string $productCode
     * @param string $code
     * @return bool
     * @throws \Doctrine\ORM\ORMException
     */
    public function disableProductVariant(string $productCode, string $code)
    {
        return $this->extensionDetector->isEnabled() && $this->setProductVariantEnabled($productCode, $code, false);
    }

    /**
     * @param string $productCode
     * @param string $code
     * @return bool
     * @throws \Doctrine\ORM\ORMException
     */
    public function enableProductVariant(string $productCode, string $code)
    {
        return $this->extensionDetector->isEnabled() && $this->setProductVariantEnabled($productCode, $code, true);
    }

    /**
     * @param string $productCode
     * @param string $code
     * @param bool $enabled
     * @return bool
     * @throws \Doctrine\ORM\ORMException
     */
    private function setProductVariantEnabled(string $productCode, string $code, bool $enabled): bool
    {
        $this->productVariantManager = $this->getClearedEntityManager($this->productVariantManager);

        $product = $this->getProduct($productCode);
        foreach ($product->getVariants() as $productVariant) {
            /** @var ProductVariant $productVariant */
            if ($productVariant->getCode() === $code) {
                $productVariant->setEnabled($enabled);
                $this->productVariantManager->persist($productVariant);
                $this->flush($this->productVariantManager);
                return true;
            }
        }
        return false;
    }

    /**
     * @param string $productCode
     * @return Product
     */
    private function getProduct(string $productCode): Product
    {
        $product = $this->productRepository->findOneBy(['code' => $productCode]);
        if (!$product instanceof Product) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed image-update! Product (code: %s) not found.',
                $productCode
            ));
        }

        return $product;
    }
}
