<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\ProductHelper;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\ORMException;
use Lifestyle\Sylius\Product\Exception\DatabaseException;
use Lifestyle\Sylius\Product\Exception\ObjectNotFoundException;
use Sylius\Component\Core\Model\Product;
use Sylius\Component\Product\Model\ProductOptionInterface;
use Sylius\Component\Product\Model\ProductVariant;
use Sylius\Component\Product\Model\ProductVariantInterface;

/**
 * Class ProductOptionSync
 *
 * Synchronize option values to current product options
 * - Removes option values of options no longer connected to product
 * - Does not add new option values for new product options
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\ProductHelper
 */
class ProductOptionSync
{
    use DatabaseTrait;

    /**
     * @var EntityRepository
     */
    private $productRepository;

    /**
     * @var EntityRepository
     */
    private $productVariantRepository;

    /**
     * @var EntityManager
     */
    private $productVariantManager;

    /**
     * ProductOptionSync constructor.
     * @param EntityRepository $productRepository
     * @param EntityRepository $productVariantRepository
     * @param EntityManager $productVariantManager
     */
    public function __construct(
        EntityRepository $productRepository,
        EntityRepository $productVariantRepository,
        EntityManager $productVariantManager
    ) {
        $this->productRepository = $productRepository;
        $this->productVariantRepository = $productVariantRepository;
        $this->productVariantManager = $productVariantManager;
    }

    /**
     * @param string $productCode
     * @throws ORMException
     * @throws DatabaseException
     */
    public function syncProductOptions(string $productCode): void
    {
        $this->productVariantManager = $this->getClearedEntityManager($this->productVariantManager);

        $product = $this->getProduct($productCode);
        $productOptionValueIds = $this->getProductOptionValueIds($product);

        // Remove option values from all variants who's options are no longer present
        foreach ($product->getVariants() as $variant) {
            /** @var ProductVariantInterface $variant */
            foreach ($variant->getOptionValues() as $optionValue) {
                if (!in_array($optionValue->getId(), $productOptionValueIds)) {
                    $variant->removeOptionValue($optionValue);
                }
            }
        }

        $this->flush($this->productVariantManager);
    }

    /**
     * @param string $productVariantCode
     * @throws DatabaseException
     * @throws ORMException
     */
    public function syncProductVariantOptions(string $productVariantCode): void
    {
        $this->productVariantManager = $this->getClearedEntityManager($this->productVariantManager);

        $productVariant = $this->getProductVariant($productVariantCode);
        $product = $productVariant->getProduct();
        if (!$product instanceof Product) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed product-option-update! No product set in variant (code: %s) not found.',
                $productVariantCode
            ));
        }
        $productOptionValueIds = $this->getProductOptionValueIds($product);

        // Remove option values who's options are no longer present
        foreach ($productVariant->getOptionValues() as $optionValue) {
            if (!in_array($optionValue->getId(), $productOptionValueIds)) {
                $productVariant->removeOptionValue($optionValue);
            }
        }

        $this->flush($this->productVariantManager);
    }

    /**
     * @param Product $product
     * @return array
     */
    private function getProductOptionValueIds(Product $product): array
    {
        $productOptionValueIds = [];
        foreach ($product->getOptions() as $option) {
            /** @var ProductOptionInterface $option */
            foreach ($option->getValues() as $optionValue) {
                $productOptionValueIds[] = $optionValue->getId();
            }
        }
        return $productOptionValueIds;
    }

    /**
     * @param string $productVariantCode
     * @return ProductVariant
     */
    private function getProductVariant(string $productVariantCode): ProductVariant
    {
        $productVariant = $this->productVariantRepository->findOneBy(['code' => $productVariantCode]);
        if (!$productVariant instanceof ProductVariant) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed product-option-update! Product-variant (code: %s) not found.',
                $productVariantCode
            ));
        }

        return $productVariant;
    }

    /**
     * @param string $productCode
     * @return Product
     */
    private function getProduct(string $productCode): Product
    {
        $product = $this->productRepository->findOneBy(['code' => $productCode]);
        if (!$product instanceof Product) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed product-option-update! Product (code: %s) not found.',
                $productCode
            ));
        }

        return $product;
    }
}
