<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\DependencyInjection;

use Lifestyle\Sylius\Product\Asset\VersionStrategy\JsonManifestVersionStrategy;
use Lifestyle\Sylius\Product\Configuration\AssetConfiguration;
use Lifestyle\Sylius\Product\Configuration\ProductConfiguration;
use Lifestyle\Sylius\Product\MessageHandler\AssetPublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\AssetUnpublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\ProductVariantPublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\ProductVariantUnpublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\TaxonUnpublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\ProductPublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\ProductUnpublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\TaxonPublishHandler;
use Lifestyle\Sylius\Product\MessageHandler\TierPriceListPublish;
use Lifestyle\Sylius\Product\MessageHandler\TierPriceListUnpublish;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;
use Symfony\Component\DependencyInjection\Loader\XmlFileLoader;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\Config\FileLocator;

/**
 * Class LifestyleSyliusProductExtension
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Lifestyle\Sylius\Product\DependencyInjection
 */
class LifestyleSyliusProductExtension extends Extension
{
    /**
     * Loads a specific configuration.
     *
     * @param array $configs
     * @param ContainerBuilder $container
     * @throws \Exception
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $configuration = $this->getConfiguration($configs, $container);
        $config = $this->processConfiguration($configuration, $configs);

        (new XmlFileLoader($container, new FileLocator(dirname(__DIR__) . '/Resources/config')))->load('services.xml');

        $this->configureAssetVersionStrategy($config, $container);
        $this->configureAssetConfiguration($config, $container);
        $this->configureProductConfiguration($config, $container);

        $this->configureMessageHandler(AssetPublishHandler::class, 'assets', 'publish', $config, $container);
        $this->configureMessageHandler(AssetUnpublishHandler::class, 'assets', 'unpublish', $config, $container);

        $this->configureMessageHandler(ProductPublishHandler::class, 'products', 'publish', $config, $container);
        $this->configureMessageHandler(ProductUnpublishHandler::class, 'products', 'unpublish', $config, $container);

        $this->configureMessageHandler(ProductVariantPublishHandler::class, 'products', 'publish', $config, $container);
        $this->configureMessageHandler(ProductVariantUnpublishHandler::class, 'products', 'unpublish', $config, $container);

        $this->configureMessageHandler(TaxonPublishHandler::class, 'taxon', 'publish', $config, $container);
        $this->configureMessageHandler(TaxonUnpublishHandler::class, 'taxon', 'unpublish', $config, $container);

        $this->configureMessageHandler(TierPriceListPublish::class, 'tierpricelist', 'publish', $config, $container);
        $this->configureMessageHandler(TierPriceListUnpublish::class, 'tierpricelist', 'unpublish', $config, $container);
    }

    /**
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureAssetVersionStrategy(array $config, ContainerBuilder $container)
    {
        if ($container->has(JsonManifestVersionStrategy::class)) {
            $definition = $container->findDefinition(JsonManifestVersionStrategy::class);
            $definition->setArgument(0, $config['assets']['json_manifest_path']);
        }
    }

    /**
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureAssetConfiguration(array $config, ContainerBuilder $container)
    {
        if ($container->has(AssetConfiguration::class)) {
            $definition = $container->findDefinition(AssetConfiguration::class);
            $definition->setArgument(0, $config['assets']['package_name']);
            $definition->setArgument(1, $config['assets']['product_to_asset_identifier']);
        }
    }

    /**
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureProductConfiguration(array $config, ContainerBuilder $container)
    {
        if ($container->has(ProductConfiguration::class)) {
            $definition = $container->findDefinition(ProductConfiguration::class);
            $definition->setArguments([
                $config['products']['fields']['extra_field_1'],
                $config['products']['fields']['extra_field_2'],
                $config['products']['fields']['extra_field_3'],
                $config['products']['fields']['extra_field_4'],
            ]);
        }
    }

    /**
     * @param string $serviceId
     * @param string $type
     * @param string $event
     * @param array $config
     * @param ContainerBuilder $container
     */
    private function configureMessageHandler(
        string $serviceId,
        string $type,
        string $event,
        array $config,
        ContainerBuilder $container
    ) {
        if (!$container->has($serviceId)) {
            return;
        }

        $definition = $container->findDefinition($serviceId);

        $responsibleEvents = isset($config[$type]['responsible_events'][$event]) ? $config[$type]['responsible_events'][$event] : [];
        foreach ($responsibleEvents as $responsibleEvent) {
            $definition->addMethodCall('addResponsibleEvent', [$responsibleEvent]);
        }
    }
}
