<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\MessageHandler;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use Lifestyle\Sylius\Product\ProductHelper\ProductAssetHelper;
use Lifestyle\Sylius\Product\ProductHelper\ProductImageHelper;
use Lifestyle\Sylius\Product\ProductHelper\ProductOptionSync;
use Lifestyle\Sylius\Product\Request\Validator\ProductPublishValidator;

/**
 * Class ProductPublishHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\Handler
 */
class ProductPublishHandler implements HandlerInterface
{
    use MessageResponsibilityTrait;
    use MessageHandlerTrait;

    /**
     * @var ProductPublishValidator
     */
    private $productValidator;

    /**
     * @var ProductImageHelper
     */
    private $productImageHelper;

    /**
     * @var ProductOptionSync
     */
    private $productOptionSync;

    /**
     * @var ProductAssetHelper
     */
    private $productAssetHelper;

    /**
     * ProductPublishHandler constructor.
     * @param ProductPublishValidator $productValidator
     * @param ProductImageHelper $productImageHelper
     * @param ProductOptionSync $productOptionSync
     * @param ProductAssetHelper $productAssetHelper
     */
    public function __construct(
        ProductPublishValidator $productValidator,
        ProductImageHelper $productImageHelper,
        ProductOptionSync $productOptionSync,
        ProductAssetHelper $productAssetHelper
    ) {
        $this->productValidator = $productValidator;
        $this->productImageHelper = $productImageHelper;
        $this->productOptionSync = $productOptionSync;
        $this->productAssetHelper = $productAssetHelper;
    }


    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     * @throws \Doctrine\ORM\ORMException
     * @throws \Lifestyle\Sylius\Product\Exception\DatabaseException
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {

        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        $this->productValidator->validate($dataCollector);

        $processed = false;
        foreach ($this->messageHandler as $handler) {
            if ($handler->isResponsible($dataCollector)) {
                $handler->createOrUpdate($dataCollector);
                $processed = true;
            }
        }

        if ($processed) {
            $this->updateImages($dataCollector);
            $this->productOptionSync->syncProductOptions($dataCollector->getItemValue('id'));
            //also add the product to asset relation here downloadable assets
            $this->productAssetHelper->updateProductAssets($dataCollector->getItemValue('id'), $dataCollector);
        }

        return $processed;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return
            in_array($dataCollector->getItemValue('event'), $this->responsibleEvents) &&
            'variant' !== $dataCollector->getItemValue('type');
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @throws \Doctrine\ORM\ORMException
     * @throws \Lifestyle\Sylius\Product\Exception\DatabaseException
     */
    private function updateImages(DataCollectorInterface $dataCollector)
    {
        $imageIds = $dataCollector->hasItem('imagesId') ? $dataCollector->getItemValue('imagesId') : [];
        $this->productImageHelper->updateProductImages(
            $dataCollector->getItemValue('id'),
            is_array($imageIds) ? $imageIds : [$imageIds]
        );
    }
}
