<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\ProductHelper;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityRepository;
use Lifestyle\Sylius\Product\Asset\VersionStrategy\JsonManifestVersionStrategy;
use Lifestyle\Sylius\Product\Configuration\AssetConfiguration;
use LifeStyle\Sylius\DataModelPlugin\Entity\ProductImage;
use Lifestyle\Sylius\Product\Exception\InvalidArgumentException;
use Lifestyle\Sylius\Product\Exception\ObjectNotFoundException;
use Symfony\Component\Asset\Packages;

/**
 * Class ImageHelper
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\ProductHelper
 */
class ImageHelper
{
    use DatabaseTrait;

    /**
     * @var AssetConfiguration
     */
    private $assetConfiguration;

    /**
     * @var JsonManifestVersionStrategy
     */
    private $assetVersionStrategy;

    /**
     * @var EntityRepository
     */
    private $imageRepository;

    /**
     * @var EntityManager
     */
    private $imageManager;

    /**
     * @var Packages
     */
    private $packages;

    /**
     * ImageHelper constructor.
     * @param AssetConfiguration $assetConfiguration
     * @param JsonManifestVersionStrategy $assetVersionStrategy
     * @param EntityRepository $imageRepository
     * @param EntityManager $imageManager
     * @param Packages $packages
     */
    public function __construct(
        AssetConfiguration $assetConfiguration,
        JsonManifestVersionStrategy $assetVersionStrategy,
        EntityRepository $imageRepository,
        EntityManager $imageManager,
        Packages $packages
    ) {
        $this->assetConfiguration = $assetConfiguration;
        $this->assetVersionStrategy = $assetVersionStrategy;
        $this->imageRepository = $imageRepository;
        $this->imageManager = $imageManager;
        $this->packages = $packages;
    }

    /**
     * @param string $resourceId
     * @throws \Doctrine\ORM\ORMException
     * @throws \Lifestyle\Sylius\Product\Exception\DatabaseException
     */
    public function updateImage(string $resourceId)
    {
        $this->assetVersionStrategy->clearCache();
        $this->imageManager = $this->getClearedEntityManager($this->imageManager);

        $resourceId = (string)$resourceId;
        if (0 === strlen($resourceId)) {
            throw new InvalidArgumentException(
                'Cannot proceed image-update! Image-Id not set in message.'
            );
        }

        $path = $this->getImagePath($resourceId);

        /** @var ProductImage[] $images */
        $images = $this->imageRepository->findBy(['resourceId' => $resourceId]);
        foreach ($images as $image) {
            $image->setPath($path);
            $this->imageManager->persist($image);
        }
        if (0 < count($images)) {
            $this->flush($this->imageManager);
        }
    }

    /**
     * @param string $resourceId
     * @throws \Doctrine\ORM\ORMException
     * @throws \Lifestyle\Sylius\Product\Exception\DatabaseException
     */
    public function removeImage(string $resourceId)
    {
        $this->assetVersionStrategy->clearCache();
        $this->imageManager = $this->getClearedEntityManager($this->imageManager);

        $resourceId = (string)$resourceId;
        if (0 === strlen($resourceId)) {
            throw new InvalidArgumentException(
                'Cannot proceed image-update! Image-Id not set in message.'
            );
        }

        /** @var ProductImage[] $images */
        $images = $this->imageRepository->findBy(['resourceId' => $resourceId]);
        foreach ($images as $image) {
            $this->imageManager->remove($image);
        }
        if (0 < count($images)) {
            $this->flush($this->imageManager);
        }
    }

    /**
     * @param string $resourceId
     * @return string
     */
    private function getImagePath(string $resourceId): string
    {
        $path = $this->packages->getUrl($resourceId, $this->assetConfiguration->getPackageName());
        if (0 === strlen($path)) {
            throw new ObjectNotFoundException(sprintf(
                'Cannot proceed image-update! Image-path (id: %s) not found.',
                $resourceId
            ));
        }

        return $path;
    }
}
