<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\MessageHandler;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use Lifestyle\Sylius\Product\ProductHelper\ProductImageHelper;
use Lifestyle\Sylius\Product\ProductHelper\ProductOptionSync;
use Lifestyle\Sylius\Product\ProductHelper\ProductVariantAssetHelper;
use Lifestyle\Sylius\Product\ProductHelper\ProductVariantExtensionDetector;
use Lifestyle\Sylius\Product\ProductHelper\TagHelper;
use Lifestyle\Sylius\Product\Request\Validator\ProductPublishValidator;

/**
 * Class ProductVariantPublishHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\Handler
 */
class ProductVariantPublishHandler implements HandlerInterface
{
    use MessageResponsibilityTrait;
    use MessageHandlerTrait;

    /**
     * @var ProductPublishValidator
     */
    private $productValidator;

    /**
     * @var ProductVariantExtensionDetector
     */
    private $productVariantExtensionDetector;

    /**
     * @var ProductImageHelper
     */
    private $productImageHelper;

    /**
     * @var ProductOptionSync
     */
    private $productOptionSync;

    /**
     * @var ProductVariantAssetHelper
     */
    private $productVariantAssetHelper;

    /**
     * @var TagHelper
     */
    private $tagHelper;

    /**
     * ProductVariantPublishHandler constructor.
     *
     * @param ProductPublishValidator         $productValidator
     * @param ProductVariantExtensionDetector $productVariantExtensionDetector
     * @param ProductImageHelper              $productImageHelper
     * @param ProductOptionSync               $productOptionSync
     * @param ProductVariantAssetHelper       $productVariantAssetHelper
     * @param TagHelper                       $tagHelper
     */
    public function __construct(
        ProductPublishValidator $productValidator,
        ProductVariantExtensionDetector $productVariantExtensionDetector,
        ProductImageHelper $productImageHelper,
        ProductOptionSync $productOptionSync,
        ProductVariantAssetHelper $productVariantAssetHelper,
        TagHelper $tagHelper
    ) {
        $this->productValidator = $productValidator;
        $this->productVariantExtensionDetector = $productVariantExtensionDetector;
        $this->productImageHelper = $productImageHelper;
        $this->productOptionSync = $productOptionSync;
        $this->productVariantAssetHelper = $productVariantAssetHelper;
        $this->tagHelper = $tagHelper;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     * @throws \Doctrine\ORM\ORMException
     * @throws \Lifestyle\Sylius\Product\Exception\DatabaseException
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        if ($this->productVariantExtensionDetector->isEnabled()) {
            if (!$dataCollector->hasItem('enabled') || !$dataCollector->getItemValue('enabled')) {
                $dataCollector = clone $dataCollector;
                $dataCollector->addItem('enabled', true);
            }
        } elseif ($dataCollector->hasItem('enabled')) {
            $dataCollector = clone $dataCollector;
            $dataCollector->removeItem('enabled');
        }

        $this->productValidator->validate($dataCollector);

        $processed = false;
        foreach ($this->messageHandler as $handler) {
            if ($handler->isResponsible($dataCollector)) {
                $handler->createOrUpdate($dataCollector);
                $processed = true;
            }
        }

        if ($processed) {
            $this->updateImages($dataCollector);
            $this->productOptionSync->syncProductVariantOptions($dataCollector->getItemValue('id'));
            //also add the product variant to asset relation here downloadable assets
            $this->productVariantAssetHelper->updateProductVariantAssets($dataCollector->getItemValue('id'), $dataCollector);
            $this->tagHelper->updateProductVariantTags($dataCollector->getItemValue('id'), $dataCollector);
        }

        return $processed;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return
            in_array($dataCollector->getItemValue('event'), $this->responsibleEvents) &&
            'variant' === $dataCollector->getItemValue('type');
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @throws \Doctrine\ORM\ORMException
     * @throws \Lifestyle\Sylius\Product\Exception\DatabaseException
     */
    private function updateImages(DataCollectorInterface $dataCollector)
    {
        $imageIds = $dataCollector->hasItem('imagesId') ? $dataCollector->getItemValue('imagesId') : [];
        $this->productImageHelper->updateProductVariantImages(
            $dataCollector->getItemValue('parentId'),
            $dataCollector->getItemValue('id'),
            is_array($imageIds) ? $imageIds : [$imageIds]
        );
    }
}
