<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\MessageHandler;

use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use Lifestyle\Sylius\Product\ProductHelper\ProductImageHelper;
use Lifestyle\Sylius\Product\ProductHelper\ProductVariantHelper;
use Lifestyle\Sylius\Product\Request\Validator\ProductUnpublishValidator;

/**
 * Class ProductVariantUnpublishHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sylius\Product\MessageHandler
 */
class ProductVariantUnpublishHandler implements HandlerInterface
{
    use MessageResponsibilityTrait;
    use MessageHandlerTrait;

    /**
     * @var ProductUnpublishValidator
     */
    private $productValidator;

    /**
     * @var ProductVariantHelper
     */
    private $productVariantHelper;

    /**
     * @var ProductImageHelper
     */
    private $productImageHelper;

    /**
     * ProductVariantUnpublishHandler constructor.
     * @param ProductUnpublishValidator $productValidator
     * @param ProductVariantHelper $productVariantHelper
     * @param ProductImageHelper $productImageHelper
     */
    public function __construct(
        ProductUnpublishValidator $productValidator,
        ProductVariantHelper $productVariantHelper,
        ProductImageHelper $productImageHelper
    ) {
        $this->productValidator = $productValidator;
        $this->productVariantHelper = $productVariantHelper;
        $this->productImageHelper = $productImageHelper;
    }

    /**
     * Disable product variant if extension is available
     * otherwise try to delete product variant. It seems like
     * there is no way to use the RESTful API because of missing
     * values in delete request to trigger a product variant update
     * request.
     *
     * Deleting product variants will fail if there is at least
     * one order containing the product variant
     *
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     * @throws \Doctrine\ORM\ORMException
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        $productCode = $dataCollector->getItemValue('parentId');
        $productVariantCode = $dataCollector->getItemValue('id');

        // If extended product-variant-model is available, disable product-variant
        if ($this->productVariantHelper->disableProductVariant($productCode, $productVariantCode)) {
            $this->productImageHelper->deleteProductVariantImages($productCode, $productVariantCode);
            return true;
        }

        // Try to delete product variant because there is no way to disable variants - hopefully it will process
        return $this->deleteProductVariant($dataCollector);
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return
            in_array($dataCollector->getItemValue('event'), $this->responsibleEvents) &&
            'variant' === $dataCollector->getItemValue('type');
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function deleteProductVariant(DataCollectorInterface $dataCollector): bool
    {
        $this->productValidator->validate($dataCollector);

        $processed = false;
        foreach ($this->messageHandler as $handler) {
            if ($handler->isResponsible($dataCollector)) {
                $handler->delete($dataCollector);
                $processed = true;
            }
        }

        return $processed;
    }
}
