<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Product\ProductHelper;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\ORMException;
use Lifestyle\Sylius\Product\Configuration\AssetConfiguration;
use Lifestyle\Sylius\Product\Exception\DatabaseException;
use Lifestyle\Sylius\Product\Entity\Product;
use Lifestyle\Sylius\ProductVariant\Entity\ProductVariant;
use Sylius\Component\Resource\Factory\Factory as ImageFactory;
use Lifestyle\DataCollector\DataCollectorInterface;

/**
 * Class ProductAssetHelper
 * @package Lifestyle\Sylius\Product\ProductHelper
 */
class ProductVariantAssetHelper
{
    use DatabaseTrait;
    use ProductEntityTrait;

    /**
     * @var AssetConfiguration
     */
    private $assetConfiguration;

    /**
     * @var ImageFactory
     */
    private $imageFactory;

    /**
     * @var EntityManager
     */
    private $imageManager;

    /**
     * @var EntityManager
     */
    private $productManager;

    /**
     * @var EntityManager
     */
    private $productVariantManager;

    /**
     * @var EntityRepository
     */
    private $productRepository;

    /**
     * @var EntityRepository
     */
    private $assetStorageRepository;

    /**
     * ProductAssetHelper constructor.
     * @param AssetConfiguration $assetConfiguration
     * @param ImageFactory $imageFactory
     * @param EntityManager $imageManager
     * @param EntityManager $productManager
     * @param EntityManager $productVariantManager
     * @param EntityRepository $productRepository
     * @param EntityRepository $assetStorageRepository
     */
    public function __construct(
        AssetConfiguration $assetConfiguration,
        ImageFactory $imageFactory,
        EntityManager $imageManager,
        EntityManager $productManager,
        EntityManager $productVariantManager,
        EntityRepository $productRepository,
        EntityRepository $assetStorageRepository
    ) {
        $this->assetConfiguration = $assetConfiguration;
        $this->imageFactory = $imageFactory;
        $this->imageManager = $imageManager;
        $this->productManager = $productManager;
        $this->productVariantManager = $productVariantManager;
        $this->productRepository = $productRepository;
        $this->assetStorageRepository = $assetStorageRepository;
    }

    /**
     * @param string $productVariantCode
     * @param DataCollectorInterface $dataCollector
     * @throws DatabaseException
     * @throws ORMException
     */
    public function updateProductVariantAssets(string $productVariantCode, DataCollectorInterface $dataCollector)
    {
        $assetIdentifier = $this->assetConfiguration->getProductToAssetIdentifier();
        $assetResourceIds = $dataCollector->hasItem($assetIdentifier) ? $dataCollector->getItemValue($assetIdentifier) : [];

        /** @var Product $product*/
        $product = $this->getProduct($this->productRepository, $dataCollector->getItemValue('parentId'));

        if(null !== $product){
            /** @var ProductVariant $productVariant*/
            $productVariant = $this->getProductVariant($productVariantCode, $product);
            if(null !== $productVariant){
                //first remove all assets from product
                if($productVariant->hasAssets()){
                    foreach ($productVariant->getAssets() as $asset){
                        $productVariant->removeAsset($asset);
                    }
                }

                //search asset
                if(!empty($assetResourceIds)){
                    foreach($assetResourceIds as $assetResourceId){
                        $assetStorageEntity = $this->assetStorageRepository->findOneBy(['resourceId' => $assetResourceId]);
                        if(null !== $assetStorageEntity){
                            $productVariant->addAsset($assetStorageEntity);
                        }
                    }
                }

                $this->productVariantManager->persist($productVariant);
                $this->flush($this->productVariantManager);
            }
        }
    }
}
