<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author    Franz Weisflug
 * @copyright 2019 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Lifestyle\Sylius\Product\ProductHelper;

use Doctrine\ORM\EntityManager;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\Product\Entity\Product;
use Lifestyle\Sylius\Product\Entity\Tag;
use Lifestyle\Sylius\ProductVariant\Entity\ProductVariant;
use Psr\Log\LoggerInterface;
use Sylius\Bundle\ResourceBundle\Doctrine\ORM\EntityRepository;
use Sylius\Component\Resource\Factory\FactoryInterface as ResourceFactory;

class TagHelper
{
    use DatabaseTrait;
    use ProductEntityTrait;

    /**
     * @var ResourceFactory
     */
    private $tagFactory;

    /**
     * @var EntityRepository
     */
    private $tagRepository;

    /**
     * @var EntityManager
     */
    private $tagManager;

    /**
     * @var EntityManager
     */
    private $productManager;

    /**
     * @var EntityRepository
     */
    private $productRepository;

    /**
     * @var EntityRepository
     */
    private $productVariantRepository;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * TagHelper constructor.
     *
     * @param ResourceFactory $tagFactory
     * @param EntityRepository $tagRepository
     * @param EntityManager $tagManager
     * @param EntityManager $productManager
     * @param EntityRepository $productRepository
     * @param EntityRepository $productVariantRepository
     * @param LoggerInterface $logger
     */
    public function __construct(
        ResourceFactory $tagFactory,
        EntityRepository $tagRepository,
        EntityManager $tagManager,
        EntityManager $productManager,
        EntityRepository $productRepository,
        EntityRepository $productVariantRepository,
        LoggerInterface $logger
    ) {
        $this->tagFactory = $tagFactory;
        $this->tagRepository = $tagRepository;
        $this->tagManager = $tagManager;
        $this->productManager = $productManager;
        $this->productRepository = $productRepository;
        $this->productVariantRepository = $productVariantRepository;
        $this->logger = $logger;
    }

    /**
     * @param array $tagList
     *
     * @throws \Doctrine\ORM\ORMException
     */
    public function create(array $tagList): void
    {
        // reconnect database if connection was lost
        $this->tagManager = $this->getClearedEntityManager($this->tagManager);
        $tags = $this->tagRepository->findBy(['name' => $tagList]);

        if (count($tagList) > count($tags)) {
            $tags = array_map(function($tag) { return $tag->getName(); }, $tags);
            $newTags = array_diff($tagList, $tags);

            foreach ($newTags as $name) {
                /** @var Tag $tag */
                $tag = $this->tagFactory->createNew();
                $tag->setName($name);

                $this->tagRepository->add($tag);
                $this->logger->info('Successfully stored tag with name: ' . $name);
            }
        }
    }

    /**
     * @param $productCode
     * @param DataCollectorInterface $dataCollector
     */
    public function updateProductTags($productCode, DataCollectorInterface $dataCollector)
    {
        /** @var Product $product */
        $product = $this->getProduct($this->productRepository, $productCode);

        $tags = $this->tagRepository->findBy(['name' => $dataCollector->getItemValue('tags')]);
        $productTags = $product->getTags();
        $removedTags = array_udiff($productTags->toArray(), $tags, function($tag1, $tag2) {
            return $tag1->getName() === $tag2->getName() ? 0 : 1;
        });

        foreach ($removedTags as $tag) {
            $product->removeTag($tag);
        }
        foreach ($tags as $tag) {
            $product->addTag($tag);
        }

        $this->productRepository->add($product);
    }

    /**
     * @param $productVariantCode
     * @param DataCollectorInterface $dataCollector
     */
    public function updateProductVariantTags($productVariantCode, DataCollectorInterface $dataCollector)
    {
        /** @var Product $product*/
        $product = $this->getProduct($this->productRepository, $dataCollector->getItemValue('parentId'));

        if (null !== $product) {
            /** @var ProductVariant $productVariant */
            $productVariant = $this->getProductVariant($productVariantCode, $product);
            if (null !== $productVariant) {
                $tags = $this->tagRepository->findBy(['name' => $dataCollector->getItemValue('tags')]);
                $productVariantTags = $productVariant->getTags();
                $removedTags = array_udiff($productVariantTags->toArray(), $tags, function($tag1, $tag2) {
                    return $tag1->getName() === $tag2->getName() ? 0 : 1;
                });

                foreach ($removedTags as $tag) {
                    $productVariant->removeTag($tag);
                }
                foreach ($tags as $tag) {
                    $productVariant->addTag($tag);
                }

                $this->productVariantRepository->add($productVariant);
            }
        }
    }
}
