<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Sylius\ProductVariant\Provider;

use Sylius\Component\Core\Calculator\ProductVariantPriceCalculatorInterface;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Core\Model\ProductVariantInterface;
use Sylius\Component\Core\Provider\ProductVariantsPricesProviderInterface;
use Sylius\Component\Product\Model\ProductOptionValueInterface;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Class ProductVariantsPricesProvider
 *
 * @package Lifestyle\Sylius\ProductVariant\Provider
 */
class ProductVariantsPricesProvider implements ProductVariantsPricesProviderInterface
{
    /**
     * @var ProductVariantPriceCalculatorInterface
     */
    private $productVariantPriceCalculator;
    /**
     * @var TranslatorInterface
     */
    private $translator;

    /**
     * ProductVariantsPricesProvider constructor.
     *
     * @param ProductVariantPriceCalculatorInterface $productVariantPriceCalculator
     */
    public function __construct(
        ProductVariantPriceCalculatorInterface $productVariantPriceCalculator,
        TranslatorInterface $translator
    ) {
        $this->productVariantPriceCalculator = $productVariantPriceCalculator;
        $this->translator = $translator;
    }

    /**
     * @param ProductInterface $product
     * @param ChannelInterface $channel
     *
     * @return array
     */
    public function provideVariantsPrices(ProductInterface $product, ChannelInterface $channel): array
    {
        $variantsPrices = [];

        /** @var ProductVariantInterface $variant */
        foreach ($product->getVariants() as $variant) {
            if ($variant->isEnabled()) {
                $variantsPrices[$variant->getId()] = $this->constructOptionsMap($variant, $channel);
            }
        }

        return $variantsPrices;
    }

    /**
     * @param ProductVariantInterface $variant
     * @param ChannelInterface        $channel
     *
     * @return array
     */
    private function constructOptionsMap(ProductVariantInterface $variant, ChannelInterface $channel): array
    {
        $optionMap = [];

        /** @var ProductOptionValueInterface $option */
        foreach ($variant->getOptionValues() as $option) {
            $optionMap[$option->getOptionCode()] = $option->getCode();
        }

        $optionMap['value'] = $this->productVariantPriceCalculator->calculate($variant, ['channel' => $channel]);
        $optionMap['reference'] = $variant->getProduct()->getId();
        // add variant sku if available
        if (method_exists($variant, 'getSku')) {
            $optionMap['articlenumber'] = $variant->getSku();
        }
        // add variant availability indicator
        if (method_exists($variant, 'getAvailabilityIndicator')) {
            $optionMap['availabilityIndicator'] = $this->translator->trans($variant->getAvailabilityIndicator());
        }

        return $optionMap;
    }
}
