<?php

/**
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\Update;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\Update
 */
class Handler
{
    const API_UPDATE_URI = '/api/v1/product-options/%s';
    const CLIENT_EXCEPTION_MESSAGE = 'Could not update product option object!';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     * @param Mapper $mapper
     */
    public function __construct(
        SerializerInterface $serializer,
        LoggerInterface $logger,
        ClientInterface $client,
        Mapper $mapper
    ) {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
        $this->mapper = $mapper;
    }


    /**
     * @param DataCollectorInterface $collector
     * @return bool
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function update(DataCollectorInterface $collector)
    {
        $productOptionRequest = $this->mapper->mapRequestModel($collector);
        $options = [
            'body' => $this->serializer->serialize($productOptionRequest, 'json')
        ];
        $uri = $this->buildUri($productOptionRequest->getCode());

        $this->logger->info(sprintf('Send PATCH request %s with body: %s', $uri, $options['body']));

        try {
            $response = $this->client->request('PATCH', $uri, $options);
        } catch (ClientException $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getResponse()->getBody()->getContents(),
                $exception->getCode()
            );
        } catch (\Exception $exception) {
            throw new RestRequestFailedException(
                self::CLIENT_EXCEPTION_MESSAGE . ' - ' . $exception->getMessage(),
                $exception->getCode()
            );
        }
        return 204 === $response->getStatusCode();
    }

    /**
     * @param $productOptionCode
     * @return string
     */
    private function buildUri($productOptionCode)
    {
        return sprintf(self::API_UPDATE_URI, $productOptionCode);
    }
}
