<?php

/**
 * Class OAuthMiddlewareAdapterTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Sylius\RestApiClientBundle\Helper\Guzzle;

use GuzzleHttp\ClientInterface;
use Lifestyle\Sylius\RestApiClientBundle\Helper\Guzzle\OAuthMiddlewareAdapter;
use Lifestyle\Sylius\RestApiClientBundle\Persistence\PersistenceInterface;
use PHPUnit\Framework\TestCase;
use Sainsburys\Guzzle\Oauth2\AccessToken;
use Sainsburys\Guzzle\Oauth2\GrantType\GrantTypeInterface;
use Sainsburys\Guzzle\Oauth2\GrantType\RefreshTokenGrantTypeInterface;

/**
 * Class OAuthMiddlewareAdapterTest
 * @package Tests\Lifestyle\Sylius\RestApiClientBundle\Helper\Guzzle
 */
class OAuthMiddlewareAdapterTest extends TestCase
{
    public function testLoadValidAccessToken()
    {
        $persistenceLayer = $this->getMockBuilder(PersistenceInterface::class)
            ->setMethods(['load', 'persist'])
            ->getMock();

        $tokenData = ['token' => md5(microtime() . rand(0, 10000)), 'expires' => time() + 3600, 'type' => 'unit_test_type'];

        $persistenceLayer->expects($this->once())
            ->method('load')
            ->willReturn($tokenData);
        $persistenceLayer->expects($this->never())
            ->method('persist');

        $client = $this->getMockBuilder(ClientInterface::class)->getMock();
        $grantType = $this->getMockBuilder(GrantTypeInterface::class)->getMock();
        $refreshToken = $this->getMockBuilder(RefreshTokenGrantTypeInterface::class)->getMock();

        $middleware = new OAuthMiddlewareAdapter($persistenceLayer, $client, $grantType, $refreshToken);

        $accessToken = $middleware->getAccessToken();

        $this->assertSame($tokenData['token'], $accessToken->getToken());
        $this->assertSame($tokenData['expires'], $accessToken->getExpires()->getTimestamp());
        $this->assertSame($tokenData['type'], $accessToken->getType());
    }

    public function testLoadExpiredAccessTokenAndPersistNewOne()
    {
        $persistenceLayer = $this->getMockBuilder(PersistenceInterface::class)
            ->setMethods(['load', 'persist'])
            ->getMock();

        $persistenceLayer->expects($this->never())
            ->method('load');

        $expiredTokenData = ['token' => md5(microtime() . rand(0, 10000)), 'expires' => time() - 3600, 'type' => 'unit_test_type'];
        $persistenceLayer->expects($this->once())
            ->method('persist')
            ->with(OAuthMiddlewareAdapter::PERSISTENCE_KEY, $expiredTokenData);

        $client = $this->getMockBuilder(ClientInterface::class)->getMock();
        $grantType = $this->getMockBuilder(GrantTypeInterface::class)->getMock();
        $refreshToken = $this->getMockBuilder(RefreshTokenGrantTypeInterface::class)->getMock();

        /** @var OAuthMiddlewareAdapter|\PHPUnit_Framework_MockObject_MockObject $middleware */
        $middleware = $this->getMockBuilder(OAuthMiddlewareAdapter::class)
            ->setConstructorArgs([$persistenceLayer, $client, $grantType, $refreshToken])
            ->setMethods(['acquireAccessToken', 'hasAccessToken'])
            ->getMock();

        $middleware->expects($this->any())
            ->method('hasAccessToken')
            ->willReturn(true);

        $middleware->expects($this->once())
            ->method('acquireAccessToken');

        // Fake accessToken property to force acquirement and persistence.
        $accessTokenFake = new AccessToken($expiredTokenData['token'], $expiredTokenData['type'], ['expires' => $expiredTokenData['expires']]);
        $middleware->setAccessToken($accessTokenFake);

        $accessToken = $middleware->getAccessToken();

        $this->assertSame($expiredTokenData['token'], $accessToken->getToken());
        $this->assertSame($expiredTokenData['expires'], $accessToken->getExpires()->getTimestamp());
        $this->assertSame($expiredTokenData['type'], $accessToken->getType());
    }
}
