<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1;

use Lifestyle\DataCollector\DataCollector;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Create\Handler as CreateHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Delete\Handler as DeleteHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Get\Handler as GetHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Update\Handler as UpdateHandler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Handler;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Factory;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductOption\Version1\Handler as OptionHandler;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * Class HandlerTest
 * @package Tests\Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1
 */
class HandlerTest extends TestCase
{
    /**
     * @var Handler
     */
    private $productHandler;

    /**
     * @var CreateHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $createHandler;

    /**
     * @var DeleteHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $deleteHandler;

    /**
     * @var GetHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $getHandler;

    /**
     * @var UpdateHandler|\PHPUnit_Framework_MockObject_MockObject
     */
    private $updateHandler;

    protected function setUp()
    {
        parent::setUp();

        $this->createHandler = $this->getMockBuilder(CreateHandler::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->deleteHandler = $this->getMockBuilder(DeleteHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['deleteByCode'])
            ->getMock();
        $this->getHandler = $this->getMockBuilder(GetHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['exists', 'getNumberOfVariantsByCode'])
            ->getMock();
        $this->updateHandler = $this->getMockBuilder(UpdateHandler::class)
            ->disableOriginalConstructor()
            ->setMethods(['update'])
            ->getMock();

        $serviceFactory = new Factory($this->createHandler, $this->deleteHandler, $this->getHandler, $this->updateHandler);
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $optionHandler = $this->getMockBuilder(OptionHandler::class)->disableOriginalConstructor()->getMock();

        $this->productHandler = new Handler($serviceFactory, $logger, $optionHandler);
    }

    public function testIsResponsible()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'Product');
        $dataCollector->addItem('type', 'object');

        $this->assertTrue($this->productHandler->isResponsible($dataCollector));
    }

    public function testIsNotResponsible()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('className', 'SomeOtherClass');
        $dataCollector->addItem('type', 'object');

        $this->assertFalse($this->productHandler->isResponsible($dataCollector));

        $dataCollector->addItem('className', 'Product');
        $dataCollector->addItem('type', 'variant');

        $this->assertFalse($this->productHandler->isResponsible($dataCollector));
    }

    public function testCreate()
    {
        $this->getHandler->expects($this->once())
            ->method('exists')
            ->willReturn(false);

        $this->createHandler->expects($this->once())
            ->method('create');

        $dataCollector = new DataCollector();

        $this->productHandler->createOrUpdate($dataCollector);
    }

    public function testUpdate()
    {
        $this->getHandler->expects($this->once())
            ->method('exists')
            ->willReturn(true);

        $this->getHandler->expects($this->once())
            ->method('getNumberOfVariantsByCode')
            ->willReturn(1);

        $this->updateHandler->expects($this->once())
            ->method('update');

        $dataCollector = new DataCollector();

        $this->productHandler->createOrUpdate($dataCollector);
    }

    public function testDelete()
    {
        $productCode = 'unitTest123';

        $this->getHandler->expects($this->once())
            ->method('exists')
            ->with($productCode)
            ->willReturn(true);

        $this->deleteHandler->expects($this->once())
            ->method('deleteByCode')
            ->with($productCode);

        $dataCollector = new DataCollector();
        $dataCollector->addItem('id', $productCode);

        $this->productHandler->delete($dataCollector);
    }

    public function testDoNotDelete()
    {
        $productCode = 'unitTest123';

        $this->getHandler->expects($this->once())
            ->method('exists')
            ->with($productCode)
            ->willReturn(false);

        $this->deleteHandler->expects($this->never())
            ->method('deleteByCode');

        $dataCollector = new DataCollector();
        $dataCollector->addItem('id', $productCode);

        $this->productHandler->delete($dataCollector);
    }
}
