<?php

/**
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create;

use Doctrine\Common\Collections\ArrayCollection;
use JMS\Serializer\SerializerBuilder;
use Lifestyle\DataCollector\DataCollector;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create\Model\Request\Product;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create\Mapper;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create\Model\Request\ChannelPrice;
use PHPUnit\Framework\TestCase;

/**
 * Class MapperTest
 * @package Tests\Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Create
 */
class MapperTest extends TestCase
{
    public function testMapping()
    {
        $collector = $this->prepareDataCollector();

        $mapper = new Mapper();
        $mapper->setModelClass(Product::class);
        $requestModel = $mapper->mapRequestModel($collector);

        // Assert base data
        $this->assertSame('123456', $requestModel->getCode());
        $this->assertSame(5, $requestModel->getOnHand());
        $this->assertTrue($requestModel->isTracked());
        $this->assertSame(100, $requestModel->getHeight());
        $this->assertSame(200, $requestModel->getWidth());
        $this->assertSame(300, $requestModel->getDepth());
        $this->assertSame(400, $requestModel->getWeight());
        $this->assertSame(10, $requestModel->getVersion());

        // Assert channel prices
        $this->assertCount(2, $requestModel->getChannelPricing());

        $this->assertInstanceOf(ChannelPrice::class, $requestModel->getChannelPricing()['DE_WEB']);
        $this->assertSame('DE_WEB', $requestModel->getChannelPricing()['DE_WEB']->getChannel());
        $this->assertSame(99.99, $requestModel->getChannelPricing()['DE_WEB']->getPrice());
        $this->assertSame(129.99, $requestModel->getChannelPricing()['DE_WEB']->getOriginalPrice());

        $this->assertInstanceOf(ChannelPrice::class, $requestModel->getChannelPricing()['US_WEB']);
        $this->assertSame('US_WEB', $requestModel->getChannelPricing()['US_WEB']->getChannel());
        $this->assertSame(49.99, $requestModel->getChannelPricing()['US_WEB']->getPrice());
        $this->assertSame(59.99, $requestModel->getChannelPricing()['US_WEB']->getOriginalPrice());

        // Assert translated fields
        $this->assertCount(2, $requestModel->getTranslations());

        $this->assertSame('English Name', $requestModel->getTranslations()['en_US']->getName());
        $this->assertSame('en_US', $requestModel->getTranslations()['en_US']->getLocale());

        $this->assertSame('German Name', $requestModel->getTranslations()['de_DE']->getName());
        $this->assertSame('de_DE', $requestModel->getTranslations()['de_DE']->getLocale());

        $this->assertEquals(new ArrayCollection(['code' => 'code_size', 'size' => 'size_code']), $requestModel->getOptionValues());
    }

    public function testSerialization()
    {
        $collector = $this->prepareDataCollector();

        $mapper = new Mapper();
        $mapper->setModelClass(Product::class);
        $requestModel = $mapper->mapRequestModel($collector);

        $serializer = SerializerBuilder::create()->build();
        $json = $serializer->serialize($requestModel, 'json');

        $this->assertSame(
            '{"code":"123456","tracked":true,"onHand":5,"height":100,"width":200,"depth":300,"weight":400,"version":10,"translations":{"en_US":{"name":"English Name"},"de_DE":{"name":"German Name"}},"optionValues":{"code":"code_size","size":"size_code"},"channelPricings":{"DE_WEB":{"price":"99.99","originalPrice":"129.99"},"US_WEB":{"price":"49.99","originalPrice":"59.99"}}}',
            $json
        );
    }

    /**
     * @return DataCollectorInterface
     */
    private function prepareDataCollector()
    {
        // Add base attributes
        $collector = new DataCollector();

        $collector->addItem('id', '123456');
        $collector->addItem('onHand', 5);
        $collector->addItem('tracked', true);
        $collector->addItem('taxCategory', 'DE19');
        $collector->addItem('height', 100);
        $collector->addItem('width', 200);
        $collector->addItem('depth', 300);
        $collector->addItem('weight', 400);
        $collector->addItem('version', 10);
        $collector->addItem('channels', ['unit_TEST']);

        $channelPriceDE = ['code' => 'DE_WEB', 'price' => 99.99, 'originalPrice' => 129.99];
        $channelPriceEN = ['code' => 'US_WEB', 'price' => 49.99, 'originalPrice' => 59.99];
        $collector->addItem('channelPrices', ['DE_WEB' => $channelPriceDE, 'US_WEB' => $channelPriceEN]);

        // Add translated fields
        $collector->addItem('name', 'English Name', 'en_US');
        $collector->addItem('name', 'German Name', 'de_DE');

        $collector->addItem('optionValues', ['code' => 'size', 'size' => 'code']);

        return $collector;
    }
}
