<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Get;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerBuilder;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Get\Handler;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Get\Model\Response\ProductVariant;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;

/**
 * Class HandlerTest
 * @package Tests\Lifestyle\Sylius\RestApiClientBundle\Services\ProductVariant\Version1\Get
 */
class HandlerTest extends TestCase
{
    public function testExists()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new Response(200, []),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $this->assertTrue($getHandler->exists('unitTestParentCode123', 'unitTestVariantCode123'));
    }

    public function testDoesNotExists()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("Mocked Client Exceptio: not found", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(404)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $this->assertFalse($getHandler->exists('unitTestParentCode123', 'unitTestVariantCode123'));
    }

    public function testGetRequestUnexpectedClientException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while fetching the object: Mocked Client Exception');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("Mocked Client Exception", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(999)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->exists('unitTestParentCode123', 'unitTestVariantCode123');
    }

    public function testGetRequestUnexpectedException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while fetching the object: unexpected mocked exception');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new \Exception("unexpected mocked exception"),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->exists('unitTestParentCode123', 'unitTestVariantCode123');
    }

    public function testGetByCodeThrowsNoParentCodeException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while fetching the object: Product or variant code is missing.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->exists(null, 'variantCode1234');
    }

    public function testGetByCodeThrowsNoVariantCodeException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while fetching the object: Product or variant code is missing.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $getHandler->exists('variantCode1234', null);
    }

    public function testBuildUri()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $client = $this->getMockBuilder(ClientInterface::class)->getMock();

        $updateHandler = new Handler($serializer, $logger, $client);

        $class = new \ReflectionClass(Handler::class);
        $method = $class->getMethod('buildUri');
        $method->setAccessible(true);

        $uri = $method->invoke($updateHandler, 'parentCode1234', 'productVariantCode12345');
        $this->assertSame('/api/v1/products/parentCode1234/variants/productVariantCode12345', $uri);
    }

    public function testGetByCodeWillReturnResponseModel()
    {
        $serializer = SerializerBuilder::create()->build();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new Response(
                200,
                [],
                '{
                          "id": 1234,
                          "code": "unit_test",
                          "optionValues": [],
                          "position": 0,
                          "translations": [],
                          "version": 1,
                          "onHold": 2,
                          "onHand": 3,
                          "tracked": true,
                          "channelPricings": [],
                          "_links": {
                            "self": {
                              "href": "\/api\/v1\/products\/MUG_TH\/variants\/medium-mug-theme"
                            },
                            "product": {
                              "href": "\/api\/v1\/products\/MUG_TH"
                            }
                          }
                       }'
            ),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $getHandler = new Handler($serializer, $logger, $client);
        $responseModel = $getHandler->getByCode('unitTestParentCode123', 'unitTestVariantCode123');

        $this->assertInstanceOf(ProductVariant::class, $responseModel);
        $this->assertSame(1234, $responseModel->getId());
        $this->assertSame('unit_test', $responseModel->getCode());
        $this->assertSame(0, $responseModel->getPosition());
        $this->assertSame(2, $responseModel->getOnHold());
        $this->assertSame(3, $responseModel->getOnHand());
        $this->assertTrue($responseModel->isTracked());
    }
}
