<?php

/**
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\Get;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\GeneralModels\Response\ProductAttributes;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Sylius\RestApiClientBundle\Services\ProductAttributes\Version1\Get
 */
class Handler
{
    const API_GET_URI = '/api/v1/product-attributes/%s';

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @param SerializerInterface $serializer
     * @param LoggerInterface $logger
     * @param ClientInterface $client
     */
    public function __construct(SerializerInterface $serializer, LoggerInterface $logger, ClientInterface $client)
    {
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->client = $client;
    }

    /**
     * @param $attributeCode
     * @return bool
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function exists($attributeCode)
    {
        return false !== $this->getByCode($attributeCode);
    }

    /**
     * @param $attributeCode
     * @return bool|ProductAttributes
     * @throws RestRequestFailedException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function getByCode($attributeCode)
    {
        if (!strlen($attributeCode)) {
            throw new RestRequestFailedException(
                'An unexpected error occurred while fetching the object: Product attribute code is missing.'
            );
        }

        $uri = $this->buildUri($attributeCode);
        $this->logger->info(sprintf('Send GET request %s', $uri));
        try {
            $response = $this->client->request('GET', $uri);
            /** @var ProductAttributes $productAttributes */
            $productAttributes = $this->serializer->deserialize(
                $response->getBody()->getContents(),
                ProductAttributes::class,
                'json'
            );
            return $productAttributes;
        } catch (ClientException $exception) {
            if (404 === $exception->getResponse()->getStatusCode()) {
                return false;
            }
            throw new RestRequestFailedException(
                sprintf('An unexpected error occurred while fetching the attribute: %s', $exception->getMessage()),
                $exception->getCode()
            );
        } catch (\Exception $exception) {
            throw new RestRequestFailedException(
                sprintf('An unexpected error occurred while fetching the attribute: %s', $exception->getMessage()),
                $exception->getCode()
            );
        }
    }

    /**
     * @param string $name
     * @return string
     */
    private function buildUri($name)
    {
        return sprintf(self::API_GET_URI, $name);
    }
}
