<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\RestApiClientBundle\Tests\Services\Product\Version1\Delete;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\Sylius\RestApiClientBundle\Services\Product\Version1\Delete\Handler;
use Lifestyle\Sylius\RestApiClientBundle\Exception\RestRequestFailedException;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;

/**
 * Class HandlerTest
 * @package Lifestyle\Sylius\RestApiClientBundle\Tests\Services\Product\Version1\Delete
 */
class HandlerTest extends TestCase
{
    public function testDeleteByCodeSuccess()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new Response(204, []),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $deleteHandler = new Handler($serializer, $logger, $client);
        $this->assertTrue($deleteHandler->deleteByCode('unitTest123'));
    }

    public function testDeleteRequestUnexpectedClientException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('Deleting the product has been failed!');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new ClientException("", $this->getMockBuilder(RequestInterface::class)->getMock(), new Response(999)),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $deleteHandler = new Handler($serializer, $logger, $client);
        $deleteHandler->deleteByCode('unitTest123');
    }

    public function testDeleteRequestUnexpectedException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('Deleting the product has been failed!');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new \Exception("unexpected mocked exception"),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $deleteHandler = new Handler($serializer, $logger, $client);
        $deleteHandler->deleteByCode('unitTest123');
    }

    public function testDeleteByCodeThrowsNoCodeException()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage('An unexpected error occurred while deleting the object: Product code is missing.');

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();

        // Create a mock and queue the responses.
        $mock = new MockHandler([]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $deleteHandler = new Handler($serializer, $logger, $client);
        $deleteHandler->deleteByCode(null);
    }

    public function testBuildUri()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $client = $this->getMockBuilder(ClientInterface::class)->getMock();

        $deleteHandler = new Handler($serializer, $logger, $client);

        $class = new \ReflectionClass(Handler::class);
        $method = $class->getMethod('buildUri');
        $method->setAccessible(true);

        $uri = $method->invoke($deleteHandler, 'productCode12345');
        $this->assertSame('/api/v1/products/productCode12345', $uri);
    }
}
